<?php

namespace App\Http\Controllers\Admin;

use App\Models\User;
use App\Models\Property;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Carbon;

class AgentVerificationController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:admin');
    }

    public function datatables(Request $request)
    {
        $datas = User::withCount('properties')->orderBy('id', 'desc');

        if ($request->filled('from_date') && $request->filled('to_date')) {
            $from = Carbon::parse($request->from_date)->startOfDay();
            $to = Carbon::parse($request->to_date)->endOfDay();
            $datas->whereBetween('created_at', [$from, $to]);
        }
        if ($request->filled('user_type')) {
            $datas->where('user_type', $request->user_type);
        }

        return Datatables::of($datas)
            ->addColumn('action', function (User $data) {
                return '<div class="btn-group mb-1">
                                    <button type="button" class="btn btn-primary btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                    ' . 'Actions' . '
                                    </button>
                                    <div class="dropdown-menu" x-placement="bottom-start">
                                    <a href="' . route('admin-user-show', $data->id) . '"  class="dropdown-item">' . __("Details") . '</a>
                                    <a href="' . route('admin-user-edit', $data->id) . '" class="dropdown-item" >' . __("Edit") . '</a>
                                    <a href="javascript:;" class="dropdown-item send" data-email="' . $data->email . '" data-toggle="modal" data-target="#vendorform">' . __("Send") . '</a>
                                    </div>
                                </div>';
            })
            // <a href="javascript:;" data-toggle="modal" data-target="#deleteModal" class="dropdown-item" data-href="' .  route('admin-user-delete', $data->id) . '">' . __("Delete") . '</a>

            ->addColumn('status', function (User $data) {
                $status = $data->is_agent === 1 ? __('Pending') : ($data->is_agent == 2 ? __('Approve') : __('Reject'));
                $status_sign = $data->is_agent === 1 ? 'warning' : ($data->is_agent == 2 ? __('success') : __('danger'));

                return '<div class="btn-group mb-1">
                                    <button type="button" class="btn btn-' . $status_sign . ' btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                        ' . $status . '
                                    </button>
                                    <div class="dropdown-menu" x-placement="bottom-start">
                                        <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.agent.verification.status', ['id1' => $data->id, 'id2' => 2]) . '">' . __("Approve") . '</a>
                                        <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.agent.verification.status', ['id1' => $data->id, 'id2' => 3]) . '">' . __("Reject") . '</a>
                                    </div>
                                    </div>';
            })
            ->rawColumns(['action', 'status'])
            ->toJson();
    }

    public function datatables1($id)
    {
        $properties = Property::with(['category', 'galleries'])
            ->where('user_id', $id)
            ->where('is_invest', 0)
            ->orderByDesc('updated_at')
            ->get();

        return DataTables::of($properties)
            ->addIndexColumn()
            ->editColumn('photo', function ($property) {
                $photo = $property->photo ? asset("assets/images/{$property->photo}") : asset('assets/images/noimage.png');
                return "<img src='{$photo}' alt='Property Image' width='60' height='40'>";
            })
            ->editColumn('category_id', fn($property) => $property->category?->title ?? 'Category Deleted')
            ->editColumn('price', fn($property) => $property->price ?? $property->expacted_price)
            ->editColumn('created_at', fn($property) => $property->created_at->format('d-m-Y'))
            ->editColumn('updated_at', fn($property) => $property->updated_at->format('d-m-Y'))
            ->addColumn('extra_info', function ($property) {
                $steps = [
                    ['step_1', 'Basic Details', 'info'],
                    ['step_2', 'Property Details', 'primary'],
                    ['step_3', 'Location Details', 'warning'],
                    ['step_4', 'Photos & Videos: ' . $property->galleries->count(), 'info']
                ];
                $chunks = collect($steps)->chunk(2);
                return $chunks->map(function ($chunk) use ($property) {
                    $row = $chunk->map(function ($step) use ($property) {
                        [$route, $text, $class] = $step;
                        return "<a target=\"_blank\" href='" . route("admin.properties.{$route}", $property->id) . "' class='badge bg-{$class} text-white mr-1 mb-1'>{$text}</a>";
                    })->implode('');
                    return "<div>{$row}</div>";
                })->implode('');
            })

            ->editColumn('type', fn($property) => "<span class='badge bg-info text-white'>" . ($property->looking_for === 'for_rent' ? 'Rent' : 'Sell') . "</span>")
            
            ->editColumn('status', function (Property $data) {
                switch ($data->status) {
                    case 1:
                        $status = __('Approved');
                        $status_sign = 'success';
                        break;
                    case 2:
                        $status = __('Rejected');
                        $status_sign = 'danger';
                        break;
                    default:
                        $status = __('Pending');
                        $status_sign = 'warning';
                        break;
                }
                return '<div class="btn-group mb-1">
                            <button type="button" class="btn btn-' . $status_sign . ' btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                                ' . $status . '
                            </button>
                            <div class="dropdown-menu">
                                <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.properties.status', ['id1' => $data->id, 'id2' => 1]) . '">' . __("Approved") . '</a>
                                <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.properties.status', ['id1' => $data->id, 'id2' => 0]) . '">' . __("Pending") . '</a>
                                <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.properties.status', ['id1' => $data->id, 'id2' => 2]) . '">' . __("Rejected") . '</a>
                            </div>
                        </div>';
            })
            ->rawColumns(['photo', 'type', 'extra_info', 'status'])
            ->toJson();
    }
    public function properties($id)
    {
        $user = User::with('properties')->findOrFail($id);

        $properties = Property::where('user_id', $id)->paginate(10);

        return view('admin.agents.properties.index', compact('id', 'properties'));
    }

    public function index()
    {
        return view('admin.agents.index');
    }

    public function status($id1, $id2)
    {
        $user = User::findOrFail($id1);

        if ($user->is_agent === 2) {
            $msg = __('Already approve to this agent!');
            return response()->json($msg);
        }

        if ($user->is_agent === 3) {
            $msg = __('Already rejected to this agent!');
            return response()->json($msg);
        }

        $user->is_agent = $id2;

        $user->update();
        $msg = 'Data Updated Successfully.';
        return response()->json($msg);
    }
}
