<?php

namespace App\Http\Controllers\Admin;

use Datatables;
use App\Models\Plan;
use App\Models\User;
use App\Models\Currency;
use App\Models\PlanHisory;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class PlanController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:admin');
    }

    public function datatables(Request $request)
    {
        $datas = Plan::query()->orderBy('id', 'desc');

        if ($request->filled('from_date') && $request->filled('to_date')) {
            $from = Carbon::parse($request->from_date)->startOfDay();
            $to = Carbon::parse($request->to_date)->endOfDay();
            $datas->whereBetween('created_at', [$from, $to]);
        }

        if ($request->filled('user_type')) {
            $datas->where('user_type', $request->user_type);
        }

        if ($request->filled('sale_type')) {
            $datas->where('sale_type', $request->sale_type);
        }

        if ($request->filled('property_type')) {
            $datas->where('property_type', $request->property_type);
        }

        return Datatables::of($datas)
            ->addIndexColumn()
            ->editColumn('created_at', function (Plan $data) {
                return $data->created_at->format('d M, Y');
            })
            ->editColumn('price', function (Plan $data) {
                return showAdminAmount($data->price);
            })
            ->editColumn('sale_type', function (Plan $data) {
                return ucwords(str_replace('_', ' ', $data->sale_type));
            })
            ->editColumn('user_type', function (Plan $data) {
                return ucwords($data->user_type);
            })
            ->addColumn('status', function (Plan $data) {
                $status = $data->status == 1 ? __('Activated') : __('Deactivated');
                $status_sign = $data->status == 1 ? 'success' : 'danger';

                return '<div class="btn-group mb-1">
                <button type="button" class="btn btn-' . $status_sign . ' btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                    ' . $status . '
                </button>
                <div class="dropdown-menu">
                    <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.plans.status', [$data->id, 1]) . '">' . __("Activate") . '</a>
                    <a href="javascript:;" data-toggle="modal" data-target="#statusModal" class="dropdown-item" data-href="' . route('admin.plans.status', [$data->id, 0]) . '">' . __("Deactivate") . '</a>
                </div>
            </div>';
            })
            ->addColumn('action', function (Plan $data) {
                return '<div class="btn-group mb-1">
                <button type="button" class="btn btn-primary btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                    ' . __('Actions') . '
                </button>
                <div class="dropdown-menu">
                    <a href="' . route('admin.plans.edit', $data->id) . '" class="dropdown-item">' . __("Edit") . '</a>
                    <a href="javascript:;" data-toggle="modal" data-target="#deleteModal" class="dropdown-item" data-href="' . route('admin.plans.delete', $data->id) . '">' . __("Delete") . '</a>
                </div>
            </div>';
            })
            ->rawColumns(['status', 'action'])
            ->toJson();
    }
    public function index()
    {
        return view('admin.plans.index');
    }

    public function create()
    {
        $data['currency'] = Currency::where('is_default', 1)->first();
        return view('admin.plans.create', $data);
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|unique:plans',
            'price' => 'required|numeric|min:0',
        ]);

        $data = new Plan();
        $input = $request->all();

        if ($request->attribute) {
            $input['attribute'] = json_encode($request->attribute, true);
        }
        $input['slot'] = $request->slot;
        $input['boost'] = $request->boost;
        $data->fill($input)->save();

        return redirect()->route('admin.plans.index')->with('success', __('New Plan Added Successfully.'));
    }

    public function edit($id)
    {
        $data['data'] = Plan::findOrFail($id);
        $data['currency'] = Currency::where('is_default', 1)->first();
        $data['attributes'] = $data['data']->attribute ? json_decode($data['data']->attribute, true) : [];

        return view('admin.plans.edit', $data);
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|unique:plans,title,' . $id,
            'price' => 'required|numeric|min:0',
        ]);

        $data = Plan::findOrFail($id);
        
        $input = $request->all();

        if ($request->attribute) {
            $input['attribute'] = json_encode($request->attribute, true);
        }
        $input['slot'] = $request->slot;
        $input['boost'] = $request->boost;
        $data->update($input);

        return redirect()->route('admin.plans.index')->with('success', __('Plan Updated Successfully.'));
    }

    public function status($id1, $id2)
    {
        $data = Plan::findOrFail($id1);
        $data->status = $id2;
        $data->update();

        return redirect()->route('admin.plans.index')->with('success', __('Status Updated Successfully.'));
    }

    public function bulkdelete(Request $request)
    {
        $ids = explode(",", $request->bulk_id);

        foreach ($ids as $id) {
            if ($id) {
                try {
                    $this->delete($id);
                } catch (\Throwable $th) {
                    return redirect()->route('admin.plans.index')->with('error', __('Something went wrong.'));
                }
            }
        }

        return redirect()->route('admin.plans.index')->with('success', __('Selected Plans Deleted Successfully.'));
    }

    public function destroy($id)
    {
        try {
            $this->delete($id);
            return redirect()->route('admin.plans.index')->with('success', __('Plan Deleted Successfully.'));
        } catch (\Throwable $th) {
            return redirect()->route('admin.plans.index')->with('error', __('Something went wrong.'));
        }
    }

    public function delete($id)
    {
        $data = Plan::findOrFail($id);
        $data->delete();
        return true;
    }
    public function activate(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'type' => 'required|exists:plans,id',
        ]);

        $plan = Plan::findOrFail($request->type);
        $user = User::findOrFail($request->user_id);

        // Get duration in days from plan_type
        $planType = $plan->plan_type;
        // Assign plan to user
        $user->plan_id = $plan->id;
        $user->ad_limit = $plan->post_limit;
        $user->plan_end_date = now()->addDays($planType);
        $user->save();
        // update plan history
        $planHisory = PlanHisory::create([
            'user_id' => $user->id,
            'plan_id' => $plan->id,
            'status' => 1,
            'activated_at' => now(),
            'expired_at' => now()->addDays($planType),
        ]);

        return redirect()->back()->with('success', __('Plan Activated Successfully.'));
    }
    public function packageHistory()
    {
        $plans = Plan::select('id', 'title')
            ->where('status', 1)
            ->get();
        return view('admin.plans.packages-expired', compact('plans'));
    }
    public function packageHistoryDatatable(Request $request)
    {
        $datas = PlanHisory::with(['plan', 'user'])->orderBy('updated_at', 'desc');
        if ($request->filled('from_date') || $request->filled('to_date')) {
            $from = $request->filled('from_date') ? Carbon::parse($request->from_date)->startOfDay() : null;
            $to = $request->filled('to_date') ? Carbon::parse($request->to_date)->endOfDay() : null;
            if ($from && $to) {
                $datas->whereBetween('expired_at', [$from, $to]);
            } elseif ($from) {
                $datas->where('expired_at', '>=', $from);
            } elseif ($to) {
                $datas->where('expired_at', '<=', $to);
            }
        }

        if ($request->filled('status')) {
            $datas->where('status', $request->status);
        }
        if ($request->filled('plan_id')) {
            $datas->where('plan_id', $request->plan_id);
        }
        if ($request->filled('sale_type')) {
            $datas->whereHas('plan', function ($q) use ($request) {
                $q->where('sale_type', $request->sale_type);
            });
        }
        if ($request->filled('user_type')) {
            $datas->whereHas('plan', function ($q) use ($request) {
                $q->where('user_type', $request->user_type);
            });
        }
        if ($request->filled('plan_type')) {
            $datas->whereHas('plan', function ($q) use ($request) {
                $q->where('plan_type', $request->plan_type);
            });
        }
        if ($request->filled('property_type')) {
            $datas->whereHas('plan', function ($q) use ($request) {
                $q->where('property_type', $request->property_type);
            });
        }

        return Datatables::of($datas)
            ->addIndexColumn()
            ->editColumn('plan.package_name', function ($data) {
                return $data->plan ? $data->plan->title : 'N/A';
            })
            ->editColumn('assigned_user_name', function ($data) {
                return $data->user ? $data->user->name : 'N/A';
            })
            ->editColumn('user_type', function ($data) {
                return $data->user ? ucfirst($data->user->user_type) : '';
            })
            ->editColumn('sale_type', function ($data) {
                return $data->plan ? ucwords(str_replace('_', ' ', $data->plan->sale_type)) : '';
            })
            ->editColumn('price', function ($data) {
                return $data->plan ? $data->plan->price : '';
            })
            ->editColumn('active_date', function ($data) {
                return $data->activated_at ? date('d-m-Y', strtotime($data->activated_at)) : '';
            })
            ->editColumn('expire_date', function ($data) {
                return $data->expired_at ? date('d-m-Y', strtotime($data->expired_at)) : '';
            })
            ->rawColumns(['plan.package_name', 'assigned_user_name', 'user_type', 'sale_type', 'plan_type', 'active_date', 'expire_date'])
            ->make(true);
    }
}
