<?php

namespace App\Http\Controllers\Admin;

use Yajra\DataTables\Facades\DataTables;
use App\Models\User;
use App\Models\Gallery;
use App\Models\Category;
use App\Models\Facility;
use App\Models\Location;
use App\Models\Property;
use App\Models\Attribute;
use App\Models\FloorPlan;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Classes\GeniusMailer;
use App\Models\Generalsetting;
use Illuminate\Support\Carbon;
use App\Models\RejectedProperty;
use App\Models\Facility_Distance;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;

class PropertyController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth:admin');
    }
    public function datatables(Request $request)
    {
        $datas = Property::with(['user' => function ($q) {
            $q->withCount('properties');
        }, 'category', 'galleries'])
            ->whereIsInvest(0)
            ->orderBy('updated_at', 'desc');
        $dateField = $request->input('date_type', 'created_at'); // default to created_at

        if ($request->filled('from_date') || $request->filled('to_date')) {
            $from = $request->filled('from_date') ? Carbon::parse($request->from_date)->startOfDay() : null;
            $to = $request->filled('to_date') ? Carbon::parse($request->to_date)->endOfDay() : null;

            if ($from && $to) {
                $datas->whereBetween($dateField, [$from, $to]);
            } elseif ($from) {
                $datas->where($dateField, '>=', $from);
            } elseif ($to) {
                $datas->where($dateField, '<=', $to);
            }
        }

        if ($request->filled('category_id')) {
            $datas->where('category_id', $request->category_id);
        }
        if ($request->filled('sale_id')) {
            $datas->where('looking_for', $request->sale_id);
        }
        if ($request->filled('type_id')) {
            $datas->where('type', $request->type_id);
        }
        if ($request->filled('status')) {
            $datas->where('status', $request->status);
        }
        if ($request->filled('is_primium')) {
            $datas->where('is_primium', $request->is_primium);
        }

        // Custom search for user name, email, or phone (not properties_count)
        if ($request->filled('search') && isset($request->search['value']) && $request->search['value'] != '') {
            $search = $request->search['value'];
            $datas->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhereHas('user', function ($qu) use ($search) {
                        $qu->where('name', 'like', "%{$search}%")
                            ->orWhere('email', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%");
                    });
            });
        }

        return Datatables::of($datas)
            ->editColumn('photo', function (Property $data) {
                $photo = $data->photo ? url('assets/images/' . $data->photo) : url('assets/images/noimage.png');
                $user = $data->user;

                $tooltip = '<div><strong>Property Owner Information</strong></div>';
                $tooltip .= '<span>🏗' . ($data->name ?? 'N/A') . '</span><br>';
                $tooltip .= '<span style="white-space:nowrap;">🤵 Owner Name:&nbsp;' . ($user->name ?? 'N/A') . '</span><br>';
                $tooltip .= '<div>☎ Phone : ' . ($user->phone ?? 'N/A') . '</div>';
                $tooltip .= '<div>👨 User Type : ' . ($user->user_type ?? 'N/A') . '</div>';
                $tooltip .= '<div>🏡 Total Listings : ' . ($user->properties_count ?? 0) . '</div>';

                return '<img src="' . $photo . '" alt="Image" width="60" data-toggle="tooltip" data-html="true" data-placement="top" title="' . e($tooltip) . '">';
            })
            ->addColumn('post', function (Property $data) {
                $user = $data->user;
                $info = [
                    ($user->user_type ?? 'N/A')
                        . '(<a target="_blank" href="' . route('agents.properties', $data->user_id) . '" >' . ($user->properties_count ?? 0) . '</a>)'
                ];

                return '<p class="mb-1">' . implode(', ', array_filter($info)) . '</p>';
            })
            ->editColumn('category_id', function (Property $data) {
                return $data->category ? $data->category->title : 'Category Deleted';
            })
            ->editColumn('price', function (Property $data) {
                return $data->price ?? $data->expacted_price;
            })
            ->addColumn('created_at', function (Property $data) {
                return $data->created_at->format('d-m-y');
            })
            ->addColumn('updated_at', function (Property $data) {
                return $data->updated_at->format('d-m-y');
            })
            ->addColumn('extra_info', function (Property $data) {
                $steps = [
                    ['step_1', 'Basic Details', 'info'],
                    ['step_2', 'Property Details', 'primary'],
                    ['step_3', 'Location Details', 'warning'],
                    ['step_4', 'Photos & Videos: ' . $data->galleries->count(), 'info']
                ];
                $chunks = collect($steps)->chunk(2);
                return $chunks->map(function ($chunk) use ($data) {
                    $row = $chunk->map(function ($step) use ($data) {
                        [$route, $text, $class] = $step;
                        return "<a target=\"_blank\" href='" . route("admin.properties.{$route}", $data->id) . "' class='badge bg-{$class} text-white mr-1 mb-1'>{$text}</a>";
                    })->implode('');
                    return "<div>{$row}</div>";
                })->implode('');
            })
            ->editColumn('type', function (Property $data) {
                if ($data->looking_for === 'for_rent') {
                    $label = 'Rent';
                    $labelColor = 'bg-primary';
                } elseif ($data->looking_for === 'for_sell') {
                    $label = 'Sell';
                    $labelColor = 'bg-warning';
                } elseif ($data->looking_for === 'for_pg') {
                    $label = 'PG';
                    $labelColor = 'bg-info';
                } else {
                    $label = 'N/A';
                    $labelColor = 'bg-secondary';
                }

                $is_primium = match ($data->is_primium) {
                    1 => ['text' => 'Primium', 'color' => 'bg-success'],
                    2 => ['text' => 'Expired', 'color' => 'bg-danger'],
                    default => ['text' => 'Free', 'color' => 'bg-secondary'],
                };

                return '<span class="badge ' . $labelColor . ' text-white">' . $label . '</span><br>
        <span class="badge ' . $is_primium['color'] . ' text-white mt-1">' . $is_primium['text'] . '</span>';
            })
            ->editColumn('status', function (Property $data) {
                switch ($data->status) {
                    case 1:
                        $status = __('Approved');
                        $status_sign = 'success';
                        break;
                    case 2:
                        $status = __('Rejected');
                        $status_sign = 'danger';
                        break;
                    case 3:
                        $status = __('Re-Approve');
                        $status_sign = 'danger';
                        break;
                    default:
                        $status = __('Pending');
                        $status_sign = 'warning';
                        break;
                }

                return '<div class="btn-group mb-1">
                        <button type="button" class="btn btn-' . $status_sign . ' btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                            ' . $status . '
                        </button>
                        <div class="dropdown-menu">
                            <a href="' . route('admin.properties.status', ['id1' => $data->id, 'id2' => 1]) . '" class="dropdown-item">' . __("Approve") . '</a>
                            <a href="' . route('admin.properties.status', ['id1' => $data->id, 'id2' => 0]) . '" class="dropdown-item">' . __("Pending") . '</a>
                            <a href="' . route('admin.properties.status', ['id1' => $data->id, 'id2' => 2]) . '" class="dropdown-item">' . __("Reject") . '</a>
                            <a href="' . route('admin.properties.status', ['id1' => $data->id, 'id2' => 3]) . '" class="dropdown-item">' . __("Re-Approve") . '</a>
                        </div>
                    </div>';
            })
            ->addColumn('action', function (Property $data) {
                return '<div class="btn-group mb-1">
                        <button type="button" class="btn btn-primary btn-sm btn-rounded dropdown-toggle" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                            Actions
                        </button>
                        <div class="dropdown-menu">
                            <a target="_blank" href="' . route('admin.properties.step_1', $data->id) . '" class="dropdown-item">' . __("Edit") . '</a>
                            <a href="' . route('admin.properties.delete', $data->id) . '" class="dropdown-item" onclick="return confirm(\'Are you sure you want to delete this property?\')">' . __("Delete") . '</a>
                        </div>
                    </div>';
            })
            ->rawColumns(['photo', 'post', 'type', 'extra_info', 'status', 'action'])
            ->toJson();
    }
    public function index()
    {
        $users = User::where('is_banned', 0)->get();
        $categories = Category::whereStatus(1)->orderBy('custom_order', 'ASC')->get();
        session()->forget('property_id');
        return view('admin.properties.index', compact('categories', 'users'));
    }
    public function payment_duration(Request $request, $id = null)
    {
        if ($request->looking_for == 'for_rent') {
            $input['payment_duration'] = $request->payment_duration;
        } else {
            $input['payment_duration'] = NULL;
        }
    }
    public function create($id = null) // for admin create property to specific user
    {
        $user = User::find((int) $id);
        $data['user'] = $user;
        session(['user_id' => $user->id]);

        $data['categories'] = Category::whereStatus(1)->orderBy('custom_order', 'ASC')->get();

        return view('admin.properties.step-1', $data);
    }
    public function step_1($id = null)
    {
        session()->forget('property_id');
        $property = is_numeric($id) ? Property::find((int) $id) : null;

        $data['property'] = $property;
        $data['categories'] = Category::whereStatus(1)->orderBy('custom_order', 'ASC')->get();
        return view('admin.properties.step-1', $data);
    }

    public function handleRejection(Request $request, $id = null)
    {

        if (is_null($id)) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid property ID.'
            ], 400);
        }

        $property = Property::find($id);
        if (!$property) {
            return response()->json([
                'success' => false,
                'message' => 'Property not found.'
            ], 404);
        }

        $reason = $request->input('reason');

        // If reason is null or empty, mark property as pending
        if (empty($reason)) {
            $property->status = 0; // pending
            $property->save();

            RejectedProperty::where('property_id', $property->id)->delete();

            return response()->json([
                'success' => true,
                'message' => 'No reason provided. Property marked as pending.',
                'redirect' => url()->previous()
            ]);
        }

        // Validate that 'reason' is an array (for checkboxes)
        $request->validate([
            'reason' => 'required|array',
            'reason.*' => 'string|max:255'
        ]);

        // Save rejection
        RejectedProperty::updateOrCreate(
            ['property_id' => $property->id],
            [
                'reason' => $reason,
                'user_id' => $property->user_id
            ]
        );

        // Mark as rejected
        $property->status = 2;
        $property->save();

        return response()->json([
            'success' => true,
            'message' => 'Property rejected successfully.',
            'redirect' => url()->previous()
        ]);
    }

    public function store_1(Request $request, $id = null)
    {
        $rules = [
            'type' => 'required',
            'category_id' => 'required',
            'looking_for' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        if ($id) {
            $property = Property::find($id);
            session()->forget('property_id'); // Remove session property_id if URL has id
        } else {
            // Step 2: If no $id, check session for existing property
            $property = session('property_id') ? Property::find(session('property_id')) : null;
        }
        // Step 3: If no existing property, create a new one
        if (!$property) {
            $property = new Property();
        }
        $input = $request->all();

        $property->fill($input)->save();

        // Step 5: Store newly created/edited property ID in session
        session(['property_id' => $property->id]);

        return redirect()->route('admin.properties.step_2', ['id' => $property->id]);
    }
    public function step_2($id = null)
    {
        $property = is_numeric($id) ? Property::find((int) $id) : Property::find(session('property_id'));
        $data['categories'] = Category::whereStatus(1)->orderBy('id', 'desc')->get() ?? collect();
        $data['attributes'] = Attribute::whereStatus(1)->whereName('Amenities')->orderBy('id', 'desc')->get();

        $data['data_attributes'] = $property ? ($property->attributes != null ? json_decode($property->attributes, true) : []) : [];

        $data['facilities'] = Facility::all();
        $data['facility_distances'] = $property && $property->facilityDistances ? $property->facilityDistances->map(function ($distance) {
            return [
                'facility_id' => $distance->facility_id,
                'distance' => $distance->distance,
                'distance_unit' => $distance->distance_unit,
            ];
        }) : collect();

        $data['property'] = $property;
        $data['selected_category'] = $property ? $property->category_id : null;

        // Redirecting based on category_id
        if ($property->looking_for == 'for_rent' && $property->type == 'residential') {
            if (in_array($data['selected_category'], [32, 33])) {
                return view('admin.properties.plot-rent', $data);
            }
            return view('admin.properties.residential-rent', $data);
        } elseif ($property->looking_for == 'for_sale' && $property->type == 'residential') {
            if (in_array($data['selected_category'], [32, 33])) {
                return view('admin.properties.create-plots', $data);
            } elseif (in_array($data['selected_category'], [34, 35, 36])) {
                return view('admin.properties.create-flats', $data);
            }
        } elseif ($property->looking_for == 'for_rent' && $property->type == 'commercial') {
            if (in_array($data['selected_category'], [38])) {   // for commercial rent office
                return view('admin.properties.office-rent', $data);
            } elseif (in_array($data['selected_category'], [39, 40, 41, 42, 44])) {  // for commercial rent retail, storage, industry, hospitality, other
                return view('admin.properties.commercial-rent', $data);
            } elseif (in_array($data['selected_category'], [33])) {  // for commercial rent plot
                return view('admin.properties.plot-rent', $data);
            }
        } elseif ($property->looking_for == 'for_sale' && $property->type == 'commercial') {
            if (in_array($data['selected_category'], [38])) {
                return view('admin.properties.office', $data);
            } elseif (in_array($data['selected_category'], [39])) {
                return view('admin.properties.retail', $data);
            } elseif (in_array($data['selected_category'], [40])) {
                return view('admin.properties.storage', $data);
            } elseif (in_array($data['selected_category'], [41])) {
                return view('admin.properties.industry', $data);
            } elseif (in_array($data['selected_category'], [42])) {
                return view('admin.properties.hospitality', $data);
            } elseif (in_array($data['selected_category'], [33])) {
                return view('admin.properties.plot', $data);
            } elseif (in_array($data['selected_category'], [44])) {
                return view('admin.properties.other', $data);
            }
        }
    }
    public function store_plots(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'plot_area' => 'required|numeric',
            'area_unit' => 'required',
            'plot_length' => 'nullable',
            'plot_width' => 'nullable',
            'floors_allowed' => 'required',
            'no_of_floors' => 'nullable',
            'boundary_wall' => 'required',
            'open_sides' => 'required|numeric',
            'construction_done' => 'required',
            'construction_type' => 'required',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required|string',
            'distance_units' => 'required|array',
            'ownership' => 'required',
            'price' => 'required|numeric',
        ];
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());
        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['attributes'] = !empty($input['attributes']) ? json_encode($input['attributes'], true) : json_encode([]);

        $property->fill($input)->save();
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                // Optional: avoid exact duplicate combinations
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted))
                    continue;
                $inserted[] = $key;
                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        session(['property_id' => $property->id]);
        return redirect()->route('admin.properties.step_3', ['id' => $property->id]);
    }
    public function store_resi_rent(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'no_of_bhk' => 'required',
            'no_of_bathroom' => 'required',
            'no_of_balcony' => 'required',
            'covered_parking' => 'required',
            'open_parking' => 'required',
            'furnish_type' => 'required',
            'area_unit' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'total_floors' => 'required',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'attributes.*' => 'required|array|min:1',
            'property_on_floor' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());
        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }



        $input['attributes'] = !empty($input['attributes']) ? json_encode($input['attributes'], true) : json_encode([]);

        $property->fill($input)->save();
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                // Optional: avoid exact duplicate combinations
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted))
                    continue;
                $inserted[] = $key;
                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        session(['property_id' => $property->id]);
        return redirect()->route('admin.properties.step_3', ['id' => $property->id]);
    }
    public function store_comm_rent(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'area_unit' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'total_floors' => 'required',
            'property_on_floor' => 'required',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'zone_type' => 'required',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'attributes.*' => 'required|array|min:1',
            'price' => 'required|numeric',
            'ownership' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());
        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }



        $input['attributes'] = !empty($input['attributes']) ? json_encode($input['attributes'], true) : json_encode([]);

        $property->fill($input)->save();
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                // Optional: avoid exact duplicate combinations
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted))
                    continue;
                $inserted[] = $key;
                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        session(['property_id' => $property->id]);
        return redirect()->route('admin.properties.step_3', ['id' => $property->id]);
    }
    public function store_plot_rent(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'possession_date' => 'required',
            'plot_area' => 'required|numeric',
            'area_unit' => 'required',
            'ownership' => 'required',
            'price' => 'required|numeric',
            'zone_type' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());
        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        $input = $request->all();

        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $property->fill($input)->save();

        session(['property_id' => $property->id]);
        return redirect()->route('admin.properties.step_3', ['id' => $property->id]);
    }
    public function store_flats(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'no_of_bhk' => 'required',
            'no_of_bathroom' => 'required',
            'no_of_balcony' => 'required',
            'covered_parking' => 'required',
            'open_parking' => 'required',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'attributes.*' => 'required|array|min:1',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }

        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }



        $input['attributes'] = json_encode($request->input('attributes', []), true);

        // Save property
        $property->fill($input)->save();

        // Store Facility Distances
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }

        // Store property ID in session
        session(['property_id' => $property->id]);

        return redirect()->route('admin.properties.step_3', ['id' => $property->id]);
    }
    public function store_office(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'zone_type' => 'required',
            'attributes.*' => 'required|array|min:1',
            'ownership' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }

        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['attributes'] = json_encode($request->input('attributes', []), true);

        // Save property
        $property->fill($input)->save();

        // Store Facility Distances
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }

        // Store property ID in session
        session(['property_id' => $property->id]);

        return redirect()->route('admin.properties.step_3', ['id' => $property->id]);
    }
    public function store_retail(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'zone_type' => 'required',
            'attributes.*' => 'required|array|min:1',
            'ownership' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }
        $input['attributes'] = json_encode($request->input('attributes', []), true);

        // Save property
        $property->fill($input)->save();

        // Store Facility Distances
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        // Store property ID in session
        session(['property_id' => $property->id]);

        return redirect()->route('admin.properties.step_3', ['id' => $property->id]);
    }
    public function store_storage(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'zone_type' => 'required',
            'attributes.*' => 'required|array|min:1',
        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }
        $input['attributes'] = json_encode($request->input('attributes', []), true);


        $property->fill($input)->save();

        // Store Facility Distances
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }

        // Store property ID in session
        session(['property_id' => $property->id]);

        return redirect()->route('admin.properties.step_3', ['id' => $property->id]);
    }
    public function store_hospitality(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'attributes.*' => 'required|array|min:1',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }

        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['attributes'] = json_encode($request->input('attributes', []), true);

        $property->fill($input)->save();

        // Store Facility Distances
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        session(['property_id' => $property->id]);

        return redirect()->route('admin.properties.step_3', ['id' => $property->id]);
    }
    public function store_plot(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'plot_area' => 'required|numeric',
            'area_unit' => 'required',
            'plot_length' => 'nullable',
            'plot_width' => 'nullable',
            'floors_allowed' => 'required',
            'no_of_floors' => 'nullable',
            'boundary_wall' => 'required',
            'open_sides' => 'required|numeric',
            'construction_done' => 'required',
            'construction_type' => 'required',
            'possession_status' => 'required',
            'possession_date' => 'nullable',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required|string',
            'distance_units' => 'required|array',
            'zone_type' => 'required',
            'ownership' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());
        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }
        $input['attributes'] = !empty($input['attributes']) ? json_encode($input['attributes'], true) : json_encode([]);
        $property->fill($input)->save();
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $inserted[] = $key;
                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        session(['property_id' => $property->id]);
        return redirect()->route('admin.properties.step_3', ['id' => $property->id]);
    }
    public function store_other(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'zone_type' => 'required',
            'attributes.*' => 'required|array|min:1',
            'ownership' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }
        $input['attributes'] = json_encode($request->input('attributes', []), true);
        $property->fill($input)->save();
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        session(['property_id' => $property->id]);

        return redirect()->route('admin.properties.step_3', ['id' => $property->id]);
    }
    public function step_3($id = null)
    {
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);
        $data['locations'] = Location::get();
        $data['property'] = $property;

        return view('admin.properties.create-step-3', $data);
    }
    public function store_step_3(Request $request, $id = null)
    {
        $rules = [
            'location_id' => 'required',
            'locality' => 'required|string|max:255',
        ];

        $validatedData = $request->validate($rules);

        // 🧹 Clean input values
        $building = preg_replace('/\s+,/', ',', $request->building);
        $building = preg_replace('/,\s*/', ', ', $building);

        $locality = preg_replace('/\s+,/', ',', $request->locality);
        $locality = preg_replace('/,\s*/', ', ', $locality);

        // Optional full address (you can combine values if needed)
        $address = $building . ', ' . $locality;

        $city = Location::find($request->location_id)->city;

        $property = $id
            ? Property::find($id)
            : (session('property_id') ? Property::find(session('property_id')) : null);

        if (!$property) {
            return redirect()->back()->with('error', __('Property not found.'));
        }

        // ✅ Now use sanitized variables
        $property->location_id = $request->location_id;
        $property->building = $building;
        $property->locality = $locality;
        $property->sub_locality = $request->sub_locality;
        $property->real_address = $address;
        $property->city = $city;

        $property->save();

        session(['property_id' => $property->id]);

        return redirect()
            ->route('admin.properties.step_4', ['id' => $property->id])
            ->with('success', __('Third Step Completed Successfully.'));
    }
    public function step_4($id = null)
    {
        $data['property'] = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);
        $data['re_message'] = RejectedProperty::where('property_id', $data['property']->id)->orderBy('updated_at', 'desc')->first();
        $data['deleted_images'] = $data['property']
            ? Gallery::onlyTrashed()->where('property_id', $data['property']->id)->get()
            : collect();

        $data['floor_plans'] = $data['property']
            ? FloorPlan::where('property_id', $data['property']->id)->get()
            : collect();
        $property = $data['property'];
        if ($property->looking_for == 'for_rent') {
            return view('admin.properties.rent-step-4', $data);
        } else
            return view('admin.properties.create-step-4', $data);
    }
    public function rent_store_step_4(Request $request, $id = null)
    {
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);

        $photoIsMissing = !$property?->photo && !$request->hasFile('photo');

        $request->validate([
            'photo' => ($photoIsMissing ? 'required' : 'nullable') . '|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        if ($property) {
            $input = $request->all();

            if ($request->hasFile('photo')) {
                $status = ExtensionValidation($request->file('photo'));
                if (!$status) {
                    return redirect()->back()->with('message', 'File format not supported');
                }
                $input['photo'] = handleMakeImage($request->file('photo'));
            }

            $property->fill($input)->save();

            session()->forget('property_id');
            return redirect()->route('admin.properties.index')
                ->with('success', __('All Steps Completed Successfully.'));
        }

        return redirect()->back()->with('error', __('Property not found.'));
    }
    public function store_step_4(Request $request, $id = null)
    {
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);

        $photoIsMissing = !$property?->photo && !$request->hasFile('photo');

        $request->validate([
            'photo' => ($photoIsMissing ? 'required' : 'nullable') . '|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);

        if ($property) {
            $input = $request->all();

            if ($request->hasFile('photo')) {
                $status = ExtensionValidation($request->file('photo'));
                if (!$status) {
                    return redirect()->back()->with('message', 'File format not supported');
                }
                $input['photo'] = handleMakeImage($request->file('photo'));
            }

            $property->fill($input)->save();

            if ($request->has('floor_names')) {
                $floorIds = $request->floor_id;
                $floorNames = $request->floor_names;
                $floorSizes = $request->floor_sizes;
                $floorImages = $request->file('floor_images');
                $property_id = $id;
                foreach ($floorIds as $index => $id) {
                    if (!empty($id)) {
                        $floorPlan = FloorPlan::find($id);
                        if ($floorPlan) {
                            $floorPlan->name = $floorNames[$index];
                            $floorPlan->size = $floorSizes[$index];
                            if (isset($floorImages[$index]) && $floorImages[$index]->isValid()) {
                                $floorPlan->photo = handleUpdateImage($floorImages[$index], $floorPlan->photo);
                            }
                            $floorPlan->save();
                        }
                    } else {
                        $newFloorPlan = new FloorPlan();
                        $newFloorPlan->name = $floorNames[$index];
                        $newFloorPlan->size = $floorSizes[$index];
                        $newFloorPlan->property_id = $property_id;
                        if (isset($floorImages[$index]) && $floorImages[$index]->isValid()) {
                            $newFloorPlan->photo = handleMakeImage($floorImages[$index]);
                        }

                        $newFloorPlan->save();
                    }
                }
            }
            session()->forget('property_id');
            return redirect()->route('admin.properties.index')
                ->with('success', __('All Steps Completed Successfully.'));
        }

        return redirect()->back()->with('error', __('Property not found.'));
    }
    public function status($id1, $id2)
    {
        $data = Property::findOrFail($id1);
        $data->status = $id2;
        $data->update();
        $gs = Generalsetting::findOrFail(1);
        if ($data->user_id) {
            $user = User::findOrFail($data->user_id);
        } else {
            $user = NULL;
        }
        $admin = auth()->guard('admin')->user();
        if ($user) {
            if ($gs->is_smtp == 1) {
                $data = [
                    'to' => $user->email,
                    'type' => "property approve",
                    'cname' => $user->name,
                    'oamount' => $data->price,
                    'aname' => $admin->name,
                    'aemail' => $admin->email,
                    'wtitle' => "",
                ];
                $mailer = new GeniusMailer();
                $mailer->sendAutoMail($data);
            } else {
                $to = $user->email;
                $subject = " Your property approved successfully.";
                $msg = "Hello " . $user->name . "!\nYour property approve by" . $admin->name . " successfully.\nThank you.";
                $headers = "From: " . $gs->from_name . "<" . $gs->from_email . ">";
                mail($to, $subject, $msg, $headers);
            }
        }
        $msg = 'Data Updated Successfully.';
        return redirect()->back()->with('success', 'Status updated successfully.');
    }
    public function bulkdelete(Request $request)
    {
        $ids = $request->bulk_id;
        $bulk_ids = explode(",", $ids);
        foreach ($bulk_ids as $key => $id) {
            if ($id) {
                try {
                    $this->delete($id);
                    $msg = 'Data Deleted Successfully.';
                } catch (\Throwable $th) {
                    $msg = 'Something went wrong.';
                }
            }
        }
        return redirect()->back()->with('success', 'Properties Deleted successfully.');
    }
    public function destroy($id)
    {
        try {
            $this->delete($id);
            $msg = 'Data Deleted Successfully.';
        } catch (\Throwable $th) {
            $msg = 'Something went wrong.';
        }
        return redirect()->back()->with('success', 'Data Deleted Successfully.');
    }
    public function delete($id)
    {
        $data = Property::findOrFail($id);
        @unlink('assets/images/' . $data->photo);
        $data->delete();
        return true;
    }
}
