<?php

namespace App\Http\Controllers\Admin;

use Datatables;
use App\Models\User;
use App\Models\Location;
use App\Models\Property;
use App\Models\Subscriber;
use Illuminate\Http\Request;
use App\Classes\GeniusMailer;
use App\Models\Generalsetting;
use App\Models\WhatsappGroupLog;
use App\Models\WhatsappTemplate;
use App\Services\WhatsAppService;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Cache;

class WhatsappController extends Controller
{

    public function __construct(protected WhatsAppService $whatsappservice)
    {
        $this->middleware('auth:admin');
    }

    public function datatables()
    {
        $datas = WhatsappTemplate::orderBy('id', 'desc');
        return DataTables::of($datas)
            ->addColumn('action', function (WhatsappTemplate $data) {
                return '<div class="action-list"><a class="btn btn-primary btn-sm btn-rounded" href="' . route('admin.whatsapp.edit', $data->id) . '"> <i class="fas fa-edit"></i>Edit</a></div>';
            })
            ->toJson();
    }

    public function index()
    {
        return view('admin.whatsapp.index');
    }

    public function config()
    {
        $gs = Generalsetting::findOrFail(1);
        return view('admin.whatsapp.config', compact('gs'));
    }

    public function edit($id)
    {
        $data = WhatsappTemplate::findOrFail($id);
        return view('admin.whatsapp.edit', compact('data'));
    }

    public function groupwhatsapp()
    {
        $config = Generalsetting::findOrFail(1);
        $baseQuery = Location::where('status', 1)->withCount('properties');
        $allCities = $baseQuery->orderBy('name')->get();
        // Group by top and other cities using collection filters
        $data['topCities'] = $allCities->where('is_top_city', 1)->sortBy('custom_order')->values();
        $data['otherCities'] = $allCities->where('is_top_city', 0)->values();
        $data['spacificCities'] = $allCities;

        return view('admin.whatsapp.group', compact('config', 'data'));
    }

    public function groupwhatsapppost(Request $request)
    {
        $query = User::where('is_banned', 0);

        echo '<!DOCTYPE html>
            <html>
            <head>
                <title>Not Integrated</title>
                <style>
                    body {
                        margin: 0;
                        height: 100vh;
                        display: flex;
                        justify-content: center;
                        align-items: center;
                        font-family: Arial, sans-serif;
                        background: #fff;
                    }
                    .message {
                        font-size: 24px;
                    }
                </style>
            </head>
            <body>
                <div class="message">WhatsApp API is not integrated</div>
            </body>
            </html>';
        die;

        if ($request->city_filter === 'top') {
            $query->whereHas('location', function ($q) {
                $q->where('is_top_city', 1);
            });
        } elseif ($request->city_filter === 'not_top') {
            $query->whereHas('location', function ($q) {
                $q->where('is_top_city', 0);
            });
        } elseif ($request->city_filter === 'one_city' && is_array($request->cities)) {
            $query->whereIn('location_id', $request->cities);
        }

        $users = $query->get();

        $imageUrl = null;
        if ($request->hasFile('image')) {
            $imageUrl = $request->file('image')->store('whatsapp_images', 'public');
        }

        foreach ($users as $user) {
            $phone = $user->phone;
            if ($request->message_type == 'text') {
                $this->sendWhatsAppText($phone, $request->body);
            } else {
                $this->sendWhatsAppImage($phone, $imageUrl);
            }
        }

        return response()->json(['message' => 'WhatsApp messages sent successfully.']);
    }

    public function update(Request $request, $id)
    {
        $data = WhatsappTemplate::findOrFail($id);
        $input = $request->all();
        $data->update($input);
        $msg = 'Data Updated Successfully.' . '<a href="' . route("admin.whatsapp.index") . '">View Template Lists</a>';
        return response()->json($msg);
    }

    // public function whatsappGroupMessage(Request $request)
    // {
    //     try {
    //         $query = User::where('user_type', $request->type);

    //         // Fetch and cache locations by ID
    //         $locationsById = Cache::rememberForever('locations_by_id', function () {
    //             return Location::all()->keyBy('id');
    //         });

    //         if (!empty($request->cities)) {
    //             $cityNames = collect($request->cities)->map(function ($locationId) use ($locationsById) {
    //                 return $locationsById[$locationId]->name ?? null;
    //             })->filter();

    //             $query->whereIn('city', $cityNames);
    //         }

    //         $users = $query->get();

    //         if ($users->isEmpty()) {
    //             return redirect()->back()->with('error', 'No users found for selected criteria.');
    //         }

    //         $message = $request->message_type == 'text' ? $request->body : '';
    //         $imageUrl = null;

    //         if ($request->hasFile('image')) {
    //             $imageUrl = $request->file('image')->store('whatsapp_images', 'public');
    //         }

    //         foreach ($users as $user) {
    //             $phone = $user->phone;
    //             $sendResult = null;

    //             if ($request->message_type == 'text') {
    //                 $sendResult = $this->whatsappservice->sendMessageWithTemplate($phone, $message);
    //             } elseif ($imageUrl) {
    //                 $sendResult = $this->whatsappservice->sendMessageWithTemplate($phone, $message, $imageUrl);
    //             }

    //             // Check if message was actually sent
    //             if (!$sendResult) {
    //                 return redirect()->back()->with('error', 'Failed to send message to ' . $user->phone);
    //             }

    //             // Try creating log
    //             $logCreated = WhatsappGroupLog::create([
    //                 'user_type' => $request->type,
    //                 'name' => $user->name,
    //                 'phone' => $user->phone,
    //                 'message' => $message,
    //                 'city' => $user->city,
    //             ]);

    //             if (!$logCreated) {
    //                 return redirect()->back()->with('error', 'Failed to log message for ' . $user->phone);
    //             }
    //         }

    //         return redirect()->back()->with('message', 'WhatsApp messages sent successfully to selected users.');
    //     } catch (\Exception $e) {
    //         return redirect()->back()->with('error', 'Something went wrong: ' . $e->getMessage());
    //     }
    // }
    // public function downloadWhatsappGroupLog()
    // {
    //     $logs = WhatsappGroupLog::orderBy('created_at', 'desc')->take(500)->get(['user_type', 'name', 'phone', 'message', 'created_at']);
    //     if ($logs->isEmpty()) {
    //         return redirect()->back()->with('error', 'No WhatsApp group log data found.');
    //     }
    //     $filename = 'whatsapp_group_log_' . now()->format('Ymd_His') . '.csv';
    //     $headers = [
    //         'Content-Type' => 'text/csv',
    //         'Content-Disposition' => "attachment; filename=\"$filename\"",
    //     ];
    //     $columns = ['user_type', 'name', 'phone', 'message', 'created_at'];
    //     $callback = function () use ($logs, $columns) {
    //         $file = fopen('php://output', 'w');
    //         fputcsv($file, $columns);
    //         foreach ($logs as $row) {
    //             fputcsv($file, $row->only($columns));
    //         }
    //         fclose($file);
    //     };
    //     return response()->stream($callback, 200, $headers);
    // }

    public function whatsappGroupMessage(Request $request)
    {
        try {
            // Get filtered users based on properties in selected cities
            $users = $this->getFilteredUsers($request);

            if ($users->isEmpty()) {
                return redirect()->back()->with('error', 'No users found for selected criteria.');
            }

            $message = $request->message_type == 'text' ? $request->body : '';
            $imageUrl = null;

            if ($request->hasFile('image')) {
                $imageUrl = $request->file('image')->store('whatsapp_images', 'public');
            }

            $successCount = 0;
            $failedNumbers = [];

            foreach ($users as $user) {
                $rawPhone = preg_replace('/\D/', '', $user->phone); // remove non-digits

                if (strlen($rawPhone) === 10) {
                    $phone = '91' . $rawPhone;
                } elseif (strlen($rawPhone) === 12 && substr($rawPhone, 0, 2) === '91') {
                    $phone = $rawPhone;
                } else {
                    // Invalid phone format
                    $failedNumbers[] = $user->phone;
                    continue;
                }
                $sendResult = false;

                try {
                    if ($request->message_type == 'text') {
                        $sendResult = $this->whatsappservice->sendMessageWithTemplate($phone, $message);
                    } elseif ($imageUrl) {
                        $sendResult = $this->whatsappservice->sendMessageWithTemplate($phone, $message, $imageUrl);
                    }

                    if ($sendResult) {
                        WhatsappGroupLog::create([
                            'user_type' => $request->type,
                            'name' => $user->name,
                            'phone' => $user->phone,
                            'message' => $message,
                            'city' => $user->city,
                            'status' => 'sent'
                        ]);
                        $successCount++;
                    } else {
                        $failedNumbers[] = $user->phone;
                    }
                } catch (\Exception $e) {
                    $failedNumbers[] = $user->phone;
                    continue;
                }
            }

            $message = "WhatsApp messages sent successfully to {$successCount} users.";
            if (!empty($failedNumbers)) {
                $message .= " Failed to send to: " . implode(', ', array_slice($failedNumbers, 0, 5));
                if (count($failedNumbers) > 5) {
                    $message .= " and " . (count($failedNumbers) - 5) . " more";
                }
            }

            return redirect()->back()->with('message', $message);
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Something went wrong: ' . $e->getMessage());
        }
    }

    // Handle download of filtered user data
    public function downloadWhatsappGroupData(Request $request)
    {
        try {
            $userType = $request->type;
            $cityFilter = $request->city_filter;
            $cityIds = [];

            if ($cityFilter === 'all') {
                // All locations
                $locationIds = \App\Models\Location::pluck('id')->toArray();
            } else {
                // Specific locations from cities[]
                $cityIds = $this->getCityIdsFromRequest($request);
                $locationIds = $cityIds;
            }

            if (empty($locationIds)) {
                return redirect()->back()->with('error', 'No cities selected.');
            }

            // Get all properties in selected locations
            $properties = \App\Models\Property::whereIn('location_id', $locationIds)
                ->whereNull('deleted_at')
                ->get(['user_id', 'location_id']);

            if ($properties->isEmpty()) {
                return redirect()->back()->with('error', 'No users found for selected criteria.');
            }

            // Unique user-location pairs
            $userLocationPairs = $properties->unique(function ($item) {
                return $item['user_id'] . '_' . $item['location_id'];
            });

            $userIds = $userLocationPairs->pluck('user_id')->unique()->toArray();
            $locationIdsUsed = $userLocationPairs->pluck('location_id')->unique()->toArray();

            // Get users with correct user_type
            $users = \App\Models\User::whereIn('id', $userIds)
                ->where('user_type', $userType)
                ->get()
                ->keyBy('id');

            // Get locations
            $locations = \App\Models\Location::whereIn('id', $locationIdsUsed)->get()->keyBy('id');

            // Prepare rows for CSV
            $rows = [];
            foreach ($userLocationPairs as $pair) {
                $user = $users->get($pair->user_id);
                $location = $locations->get($pair->location_id);
                if ($user && $location) {
                    $rows[] = [
                        'created_at' => $user->created_at,
                        'user_type'  => $user->user_type,
                        'name'       => $user->name,
                        'phone'      => $user->phone,
                        'city'       => $location->name,
                    ];
                }
            }

            if (empty($rows)) {
                return redirect()->back()->with('error', 'No users found for selected criteria.');
            }

            $filename = 'whatsapp_users_' . now()->format('Ymd_His') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=\"$filename\"",
            ];

            $columns = ['created_at', 'user_type', 'name', 'phone', 'city'];

            $callback = function () use ($rows, $columns) {
                $file = fopen('php://output', 'w');
                fputcsv($file, $columns);

                foreach ($rows as $row) {
                    fputcsv($file, [
                        $row['created_at'],
                        $row['user_type'],
                        $row['name'],
                        $row['phone'],
                        $row['city']
                    ]);
                }

                fclose($file);
            };

            return response()->stream($callback, 200, $headers);
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to generate download: ' . $e->getMessage());
        }
    }

    private function getFilteredUsers(Request $request)
    {
        $userType = $request->type;
        $cityFilter = $request->city_filter;
        $cityIds = [];

        if ($cityFilter === 'all') {
            $locationIds = \App\Models\Location::pluck('id')->toArray();
        } else {
            $cityIds = $this->getCityIdsFromRequest($request);
            $locationIds = $cityIds;
        }

        if (empty($locationIds)) {
            return collect(); // return empty collection
        }

        $properties = \App\Models\Property::whereIn('location_id', $locationIds)
            ->whereNull('deleted_at')
            ->get(['user_id', 'location_id']);

        $userLocationPairs = $properties->unique(function ($item) {
            return $item['user_id'] . '_' . $item['location_id'];
        });

        $userIds = $userLocationPairs->pluck('user_id')->unique()->toArray();
        $locationIdsUsed = $userLocationPairs->pluck('location_id')->unique()->toArray();

        $users = \App\Models\User::whereIn('id', $userIds)
            ->where('user_type', $userType)
            ->get()
            ->map(function ($user) use ($locationIdsUsed) {
                $location = \App\Models\Location::find($user->location_id); // assuming user has location_id
                $user->city = $location ? $location->name : null;
                return $user;
            });

        return $users;
    }

    // Download logs (unchanged)
    public function downloadWhatsappGroupLog()
    {
        $logs = WhatsappGroupLog::orderBy('created_at', 'desc')->take(500)->get(['user_type', 'name', 'phone', 'message', 'created_at']);

        if ($logs->isEmpty()) {
            return redirect()->back()->with('error', 'No WhatsApp group log data found.');
        }

        $filename = 'whatsapp_group_log_' . now()->format('Ymd_His') . '.csv';
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename=\"$filename\"",
        ];

        $columns = ['user_type', 'name', 'phone', 'message', 'created_at'];

        $callback = function () use ($logs, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);
            foreach ($logs as $row) {
                fputcsv($file, $row->only($columns));
            }
            fclose($file);
        };
        return response()->stream($callback, 200, $headers);
    }

    /**
     * Helper method to get city IDs from request
     */
    private function getCityIdsFromRequest($request)
    {
        if (empty($request->cities)) {
            return [];
        }

        if (is_array($request->cities)) {
            return $request->cities;
        }

        if (is_string($request->cities)) {
            return json_decode($request->cities, true) ?? [];
        }

        return [];
    }
}
