<?php

namespace App\Http\Controllers\Agent;

use Datatables;
use App\Models\Plan;
use App\Models\Room;
use App\Models\Country;
use App\Models\Gallery;
use App\Models\Category;
use App\Models\Facility;
use App\Models\Location;
use App\Models\Property;
use App\Models\Schedule;
use App\Models\Attribute;
use App\Models\FloorPlan;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use App\Models\RejectedProperty;
use App\Models\Facility_Distance;
use Illuminate\Support\Facades\DB;
use function Laravel\Prompts\error;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class PropertyController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }
    public function datatables(Request $request)
    {
        $saleType = $request->sale_id;

        if ($request->status != 'all') {
            $datas = Property::whereUserId(auth()->id())
                ->whereStatus($request->status)
                ->orderBy('id', 'desc');

            if ($saleType) {
                $datas = $datas->where('looking_for', $saleType);
            }
        } else {
            $datas = Property::whereUserId(auth()->id())
                ->orderBy('id', 'desc');

            if ($saleType) {
                $datas = $datas->where('looking_for', $saleType);
            }
        }

        return Datatables::of($datas)
            ->editColumn('photo', function (Property $data) {
                $photo = $data->photo ? url('assets/images/' . $data->photo) : url('assets/images/noimage.png');
                return '<img src="' . $photo . '" alt="Image" width="100px">';
            })
            ->editColumn('category_id', function (Property $data) {
                return $data->category != NULL ? $data->category->title : 'Category Deleted';
            })
            ->editColumn('location_id', function (Property $data) {
                return $data->location != NULL ? $data->location->name : 'Step 2 is Incomplete';
            })
            ->editColumn('status', function (Property $data) {
                switch ($data->status) {
                    case 2:
                        $rejection = RejectedProperty::where('property_id', $data->id)->first();
                        $reason = $rejection ? e(is_array($rejection->reason) ? json_encode($rejection->reason) : $rejection->reason) : 'No reason provided';
                        return '<span class="badge bg-danger text-white">Rejected</span>
                        <br>
                        <a href="javascript:void(0);" class="text-danger view-rejection-reason" data-reason="' . $reason . '">
                            <small><i class="ti-info-alt"></i> View Reason</small>
                        </a>';

                    case 0:
                        return '<span class="badge bg-warning text-white">Pending</span>';
                    case 1:
                        return '<span class="badge bg-success text-white">Approved</span>';
                    case 3:
                        return '<span class="badge bg-danger text-white">Re-Approve</span>';
                    default:
                        return '<span class="badge bg-danger text-white">Expired</span>';
                }
            })
            ->addColumn('created_at', function (Property $data) {
                return $data->created_at->format('d-m-y');
            })
            ->addColumn('updated_at', function (Property $data) {
                return $data->updated_at->format('d-m-y');
            })
            ->editColumn('type', function (Property $data) {
                if ($data->looking_for == 'for_sell') {
                    return '<span class="badge bg-info text-white">Sell</span>';
                } elseif ($data->looking_for == 'for_rent') {
                    return '<span class="badge bg-info text-white">Rent</span>';
                } else {
                    return '<span class="badge bg-info text-white">PG</span>';
                }
            })
            ->addColumn('action', function (Property $data) {
                $property = $data;
                $routes = [
                    'for_pg' => 'pg.details',
                    'for_rent' => 'rent.details',
                    'for_sell' => 'buy.details',
                    'plot' => 'plot.details',
                    'commercial' => 'commercial.details',
                ];
                $propertyUrl = isset($routes[$property->looking_for])
                    ? route($routes[$property->looking_for], $property->slug ?? '')
                    : '#';
                $view_btn = '<a class="dropdown-item" href="' . $propertyUrl . '" target="_blank">View</a>';

                $viewLink = $data->status == 1
                    ? $view_btn
                    : '';

                // Show make premium button only if status == 1
                $makePremiumBtn = '';
                if ($data->status == 1) {
                    $makePremiumBtn = '<button type="button" class="btn btn-warning mt-2 make-premium-btn" style="font-size:11px;" data-id="' . $data->id . '" data-toggle="modal" data-target="#makePremiumModal">' . __("Make Premium") . '</button>';
                }

                return '<div class="btn-group mb-1 d-flex flex-column align-items-start">
            <button type="button" class="btn btn-primary dropdown-toggle mb-1" style="font-size:11px;" data-toggle="dropdown" aria-haspopup="true" aria-expanded="false">
                ' . __("Actions") . '
            </button>
            <div class="dropdown-menu" x-placement="bottom-start">
                <a href="' . route('agent.properties.step_1', $data->id) . '" class="dropdown-item">' . __("Edit") . '</a>
                <a href="javascript:;" data-toggle="modal" data-target="#deleteModal" class="dropdown-item" data-href="' . route('agent.properties.delete', $data->id) . '">' . __("Delete") . '</a>
                ' . $viewLink . '
            </div>'
                    . $makePremiumBtn .
                    '</div>';
            })
            ->rawColumns(['photo', 'type', 'status', 'extra_info', 'action'])
            ->toJson();
    }
    public function index()
    {
        $user = Auth::user();
        $totalCredit = $user->plan_id ? (Plan::find($user->plan_id)->post_limit ?? 0) : 0;
        $availableCredit = $user->ad_limit ?? 0;
        session()->forget('property_id');
        return view('agent.properties.index', compact('totalCredit', 'availableCredit'));
    }
    public function pending()
    {
        return view('agent.properties.pending');
    }
    public function approved()
    {
        $user = Auth::user();
        $totalCredit = $user->plan_id ? (Plan::find($user->plan_id)->post_limit ?? 0) : 0;
        $availableCredit = $user->ad_limit ?? 0;
        return view('agent.properties.approved', compact('totalCredit', 'availableCredit'));
    }
    public function step_1($id = null)
    {
        $property = is_numeric($id) ? Property::find((int) $id) : Property::find(session('property_id'));
        $data['property'] = $property;
        $data['categories'] = Category::whereStatus(1)->orderBy('custom_order', 'ASC')->get();
        return view('agent.properties.step-1', $data);
    }
    public function store_1(Request $request, $id = null)
    {
        $rules = [
            'type' => 'required',
            'category_id' => 'required',
            'looking_for' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = auth()->user();

        if ($id) {
            $property = Property::find($id);
            session()->forget('property_id');
        } else {
            // Step 2: If no $id, check session for existing property
            $property = session('property_id') ? Property::find(session('property_id')) : null;
        }

        // Step 3: If no existing property, create a new one
        if (!$property) {
            $property = new Property();
        }

        // Step 4: Assign data & save
        $input = $request->all();
        $input['user_id'] = $user->id;
        $property->fill($input)->save();

        // Step 5: Store newly created/edited property ID in session
        session(['property_id' => $property->id]);
        return redirect()->route('agent.properties.step_2', ['id' => $property->id]);
    }
    public function step_2($id = null)
    {
        $property = is_numeric($id) ? Property::with('floorPlans')->find((int) $id) : Property::with('floorPlans')->find(session('property_id'));
        $data['categories'] = Category::whereStatus(1)->orderBy('id', 'desc')->get() ?? collect();
        $data['attributes'] = Attribute::whereStatus(1)
            ->with(['options' => function($query) {
                $query->where('status', 1); // sirf active options
            }])
            ->orderBy('id', 'desc')
            ->get();

        $data['data_attributes'] = $property ? ($property->attributes != null ? json_decode($property->attributes, true) : []) : [];

        $data['facilities'] = Facility::all();
        $data['facility_distances'] = $property && $property->facilityDistances ? $property->facilityDistances->map(function ($distance) {
            return [
                'facility_id' => $distance->facility_id,
                'distance' => $distance->distance,
                'distance_unit' => $distance->distance_unit,
            ];
        }) : collect();

        $data['property'] = $property;
        $data['selected_category'] = $property ? $property->category_id : null;

        if ($property->looking_for == 'for_rent' && $property->type == 'residential') {

            if (in_array($data['selected_category'], [32, 33])) {
                return view('agent.properties.plot-rent', $data);
            }
            return view('agent.properties.residential-rent', $data);
        } elseif ($property->looking_for == 'for_sell' && $property->type == 'residential') {
            //  dd('residential-for_sell');
            if (in_array($data['selected_category'], [32, 33])) {
                return view('agent.properties.create-plots', $data);
            } elseif (in_array($data['selected_category'], [34, 35, 36])) {
                if (auth()->user()->user_type == 'builder/developer') {
                    return view('agent.properties.builder-plan', $data);
                }
                return view('agent.properties.create-flats', $data);
            }
        } elseif ($property->looking_for == 'for_pg' && $property->type == 'residential') {
            return view('agent.properties.pg-plan', $data);
        } elseif ($property->looking_for == 'for_rent' && $property->type == 'commercial') {
            // dd('commercial-for_rent');
            if (in_array($data['selected_category'], [38])) {   // for commercial rent office
                return view('agent.properties.office-rent', $data);
            } elseif (in_array($data['selected_category'], [39, 40, 41, 42, 44])) {
                return view('agent.properties.commercial-rent', $data);
            } elseif (in_array($data['selected_category'], [33])) {  // for commercial rent plot
                return view('agent.properties.plot-rent', $data);
            }
        } elseif ($property->looking_for == 'for_sell' && $property->type == 'commercial') {
            // dd('commercial-for_sell');
            if (in_array($data['selected_category'], [38])) {
                return view('agent.properties.office', $data);
            } elseif (in_array($data['selected_category'], [39])) {
                return view('agent.properties.retail', $data);
            } elseif (in_array($data['selected_category'], [40])) {
                return view('agent.properties.storage', $data);
            } elseif (in_array($data['selected_category'], [41])) {
                return view('agent.properties.industry', $data);
            } elseif (in_array($data['selected_category'], [42])) {
                return view('agent.properties.hospitality', $data);
            } elseif (in_array($data['selected_category'], [33])) {
                return view('agent.properties.plot', $data);
            } elseif (in_array($data['selected_category'], [44])) {
                return view('agent.properties.other', $data);
            }
        }
    }
    public function store_plots(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'plot_area' => 'required|numeric',
            'area_unit' => 'required',
            'plot_length' => 'nullable',
            'plot_width' => 'nullable',
            'floors_allowed' => 'required',
            'no_of_floors' => 'nullable',
            'boundary_wall' => 'required',
            'open_sides' => 'required|numeric',
            'construction_done' => 'required',
            'construction_type' => 'required',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required|string',
            'distance_units' => 'required|array',
            'ownership' => 'required',
            'price' => 'required|numeric',
        ];
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            // $allErrors = $validator->errors()->all();
            // echo "<pre>";
            // print_r($allErrors);
            // echo "</pre>";            
            return back()->withErrors($validator)->withInput();
        }
        $user = auth()->user();
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());
        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['status'] = 0;
        $input['user_id'] = $user->id;
        $input['attributes'] = !empty($input['attributes']) ? json_encode($input['attributes'], true) : json_encode([]);
        if ($user->plan) {
            $input['expire_date'] = $property->exists && $property->expire_date ? $property->expire_date : Carbon::now()->addDays($user->plan->post_duration);
        }
        $property->fill($input)->save();
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                // Optional: avoid exact duplicate combinations
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted))
                    continue;
                $inserted[] = $key;
                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        session(['property_id' => $property->id]);
        return redirect()->route('agent.properties.step_3', ['id' => $property->id]);
    }
    public function store_office(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'zone_type' => 'required',
            'attributes.*' => 'required|array|min:1',
            'ownership' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = auth()->user();

        // Find or create the property
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }

        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['status'] = 0;
        $input['user_id'] = $user->id;
        $input['attributes'] = json_encode($request->input('attributes', []), true);

        // Set Expiration Date based on User Plan
        if ($user->plan) {
            $input['expire_date'] = $property->exists && $property->expire_date
                ? $property->expire_date
                : Carbon::now()->addDays($user->plan->post_duration);
        }

        // Save property
        $property->fill($input)->save();

        // Store Facility Distances
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }

        // Store property ID in session
        session(['property_id' => $property->id]);

        return redirect()->route('agent.properties.step_3', ['id' => $property->id]);
    }
    public function store_retail(Request $request, $id = null)
    {
        // dd($request);
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'attributes.*' => 'required|array|min:1',
            'ownership' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = auth()->user();

        // Find or create the property
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }

        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['status'] = 0;
        $input['user_id'] = $user->id;
        $input['attributes'] = json_encode($request->input('attributes', []), true);

        // Set Expiration Date based on User Plan
        if ($user->plan) {
            $input['expire_date'] = $property->exists && $property->expire_date
                ? $property->expire_date
                : Carbon::now()->addDays($user->plan->post_duration);
        }

        // Save property
        $property->fill($input)->save();

        // Store Facility Distances
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }

        // Store property ID in session
        session(['property_id' => $property->id]);

        return redirect()->route('agent.properties.step_3', ['id' => $property->id]);
    }
    public function store_storage(Request $request, $id = null)
    {
        // dd($request);
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'attributes.*' => 'required|array|min:1',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = auth()->user();
        // Find or create the property
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }

        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['status'] = 0;
        $input['user_id'] = $user->id;
        $input['attributes'] = json_encode($request->input('attributes', []), true);

        // Set Expiration Date based on User Plan
        if ($user->plan) {
            $input['expire_date'] = $property->exists && $property->expire_date
                ? $property->expire_date
                : Carbon::now()->addDays($user->plan->post_duration);
        }

        // Save property
        $property->fill($input)->save();

        // Store Facility Distances
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }

        // Store property ID in session
        session(['property_id' => $property->id]);

        return redirect()->route('agent.properties.step_3', ['id' => $property->id]);
    }
    public function store_hospitality(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'attributes.*' => 'required|array|min:1',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = auth()->user();

        // Find or create the property
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }

        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['status'] = 0;
        $input['user_id'] = $user->id;
        $input['attributes'] = json_encode($request->input('attributes', []), true);

        // Set Expiration Date based on User Plan
        if ($user->plan) {
            $input['expire_date'] = $property->exists && $property->expire_date
                ? $property->expire_date
                : Carbon::now()->addDays($user->plan->post_duration);
        }

        // Save property
        $property->fill($input)->save();

        // Store Facility Distances
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }

        // Store property ID in session
        session(['property_id' => $property->id]);

        return redirect()->route('agent.properties.step_3', ['id' => $property->id]);
    }
    public function store_plot(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'plot_area' => 'required|numeric',
            'area_unit' => 'required',
            'plot_length' => 'nullable',
            'plot_width' => 'nullable',
            'floors_allowed' => 'required',
            'no_of_floors' => 'nullable',
            'boundary_wall' => 'required',
            'open_sides' => 'required|numeric',
            'construction_done' => 'required',
            'construction_type' => 'required',
            'possession_status' => 'required',
            'possession_date' => 'nullable',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required|string',
            'distance_units' => 'required|array',
            'zone_type' => 'required',
            'ownership' => 'required',
            'price' => 'required|numeric',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $user = auth()->user();

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());
        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['status'] = 0;
        $input['user_id'] = $user->id;
        $input['attributes'] = !empty($input['attributes']) ? json_encode($input['attributes'], true) : json_encode([]);
        if ($user->plan) {
            $input['expire_date'] = $property->exists && $property->expire_date ? $property->expire_date : Carbon::now()->addDays($user->plan->post_duration);
        }
        $property->fill($input)->save();
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                // Optional: avoid exact duplicate combinations
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted))
                    continue;
                $inserted[] = $key;
                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        session(['property_id' => $property->id]);
        return redirect()->route('agent.properties.step_3', ['id' => $property->id]);
    }
    public function store_other(Request $request, $id = null)
    {
        // dd($request);
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'zone_type' => 'required',
            'attributes.*' => 'required|array|min:1',
            'ownership' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = auth()->user();

        // Find or create the property
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }

        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['status'] = 0;
        $input['user_id'] = $user->id;
        $input['attributes'] = json_encode($request->input('attributes', []), true);

        // Set Expiration Date based on User Plan
        if ($user->plan) {
            $input['expire_date'] = $property->exists && $property->expire_date
                ? $property->expire_date
                : Carbon::now()->addDays($user->plan->post_duration);
        }

        // Save property
        $property->fill($input)->save();

        // Store Facility Distances
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }

        // Store property ID in session
        session(['property_id' => $property->id]);

        return redirect()->route('agent.properties.step_3', ['id' => $property->id]);
    }
    public function store_resi_rent(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'no_of_bhk' => 'required',
            'no_of_bathroom' => 'required',
            'no_of_balcony' => 'required',
            'covered_parking' => 'required',
            'open_parking' => 'required',
            'furnish_type' => 'required',
            'area_unit' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'total_floors' => 'required',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'attributes.*' => 'required|array|min:1',
            'property_on_floor' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $user = auth()->user();
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());
        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['status'] = 0;
        $input['user_id'] = $user->id;
        $input['attributes'] = !empty($input['attributes']) ? json_encode($input['attributes'], true) : json_encode([]);
        if ($user->plan) {
            $input['expire_date'] = $property->exists && $property->expire_date ? $property->expire_date : Carbon::now()->addDays($user->plan->post_duration);
        }
        $property->fill($input)->save();
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                // Optional: avoid exact duplicate combinations
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted))
                    continue;
                $inserted[] = $key;
                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        session(['property_id' => $property->id]);
        return redirect()->route('agent.properties.step_3', ['id' => $property->id]);
    }
    public function store_comm_rent(Request $request, $id = null)
    {
        //  dd($request->possession_date);
        $rules = [
            'name' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'furnish_type' => 'required',
            'area_unit' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'total_floors' => 'required',
            'property_on_floor' => 'required',
            'located_inside' => 'required',
            'other_located_inside' => 'nullable',
            'zone_type' => 'required',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'attributes.*' => 'required|array|min:1',
            'price' => 'required|numeric',
            'ownership' => 'required',
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }
        $user = auth()->user();
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());
        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['status'] = 0;
        $input['user_id'] = $user->id;
        $input['attributes'] = !empty($input['attributes']) ? json_encode($input['attributes'], true) : json_encode([]);
        if ($user->plan) {
            $input['expire_date'] = $property->exists && $property->expire_date ? $property->expire_date : Carbon::now()->addDays($user->plan->post_duration);
        }
        $property->fill($input)->save();
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                // Optional: avoid exact duplicate combinations
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted))
                    continue;
                $inserted[] = $key;
                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        session(['property_id' => $property->id]);
        return redirect()->route('agent.properties.step_3', ['id' => $property->id]);
    }
    public function store_plot_rent(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'plot_area' => 'required|numeric',
            'plot_length' => 'required',
            'plot_width' => 'required',
            'area_unit' => 'required',
            'ownership' => 'required',
            'price' => 'required|numeric',
            'zone_type' => 'required',
        ];
        if ($request->input('possession_status') === 'future') {
            $rules['plot_possession_date'] = 'required|date';
        }
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {

            $allErrors = $validator->errors()->all();
            // echo "<pre>";
            // print_r($allErrors);
            // echo "</pre>";   die;
            return back()->withErrors($validator)->withInput();
        }
        $user = auth()->user();
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());
        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }
        $input = $request->all();

        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['status'] = 0;
        $input['user_id'] = $user->id;

        $property->fill($input)->save();

        session(['property_id' => $property->id]);
        return redirect()->route('agent.properties.step_3', ['id' => $property->id]);
    }
    public function store_flats(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'no_of_bhk' => 'required',
            'no_of_bathroom' => 'required',
            'no_of_balcony' => 'required',
            'covered_parking' => 'required',
            'open_parking' => 'required',
            'furnish_type' => 'required',
            'built_up_area' => 'required',
            'carpet_area' => 'required',
            'property_on_floor' => 'required',
            'total_floors' => 'required',
            'price_per_sqft' => 'required|numeric',
            'price' => 'required|numeric',
            'facilities' => 'required|array|min:2',
            'facilities.*' => 'required',
            'distance_units' => 'required|array',
            'attributes.*' => 'required|array|min:1',
        ];

        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = auth()->user();

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }

        // Collect input
        $input = $request->all();
        $facilities = $request->input('facilities', []);
        $distances = $request->input('distances', []);
        $distance_units = $request->input('distance_units', []);

        // Generate Unique Slug
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            // If updating, retain the slug unless the name has changed
            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }

        $input['status'] = 0;
        $input['user_id'] = $user->id;
        $input['attributes'] = json_encode($request->input('attributes', []), true);

        // Save property
        $property->fill($input)->save();

        // Store Facility Distances
        Facility_Distance::where('property_id', $property->id)->delete();
        if (
            is_array($facilities) &&
            is_array($distances) &&
            is_array($distance_units) &&
            count($facilities) === count($distances) &&
            count($facilities) === count($distance_units)
        ) {
            $inserted = [];
            foreach ($facilities as $index => $facilityId) {
                $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                if (in_array($key, $inserted)) continue;
                $inserted[] = $key;

                Facility_Distance::create([
                    'facility_id' => $facilityId,
                    'property_id' => $property->id,
                    'distance' => $distances[$index],
                    'distance_unit' => $distance_units[$index],
                ]);
            }
        }
        session(['property_id' => $property->id]);

        return redirect()->route('agent.properties.step_3', ['id' => $property->id]);
    }
    public function builder_project(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required|string|max:100',
            'sale_type' => 'required',
            'construction_status' => 'required',
            'possession_date' => 'nullable',
            'property_age' => 'nullable',
            'bhk_names' => 'required|array',
            'bhk_names.*' => 'string',
            'flats' => 'required|array',
        ];

        $customMessages = [];
        $customAttributes = [];

        foreach ($request->input('bhk_names', []) as $bhk) {
            $flatData = $request->input("flats.$bhk");
            if (empty($flatData) || !is_array($flatData)) {
                continue;
            }

            // Define rules
            $rules["flats.$bhk.no_of_bathroom"] = 'required|integer|min:1';
            $rules["flats.$bhk.no_of_balcony"] = 'required|integer|min:0';
            $rules["flats.$bhk.covered_parking"] = 'required';
            $rules["flats.$bhk.open_parking"] = 'required';
            $rules["flats.$bhk.furnish_type"] = 'required|string';
            $rules["flats.$bhk.built_up_area"] = 'required|numeric|min:0';
            $rules["flats.$bhk.carpet_area"] = 'required|numeric|min:0';
            $rules["flats.$bhk.total_floors"] = 'required|integer|min:1';
            $rules["flats.$bhk.property_on_floor"] = 'required|string';
            $rules["flats.$bhk.price_per_sqft"] = 'required|numeric|min:0';
            $rules["flats.$bhk.expected_price"] = 'required|numeric|min:0';

            // Define custom attribute names for better error messages
            $customAttributes["flats.$bhk.no_of_bathroom"] = "$bhk - No. of Bathrooms";
            $customAttributes["flats.$bhk.no_of_balcony"] = "$bhk - No. of Balcony";
            $customAttributes["flats.$bhk.covered_parking"] = "$bhk - Covered Parking";
            $customAttributes["flats.$bhk.open_parking"] = "$bhk - Open Parking";
            $customAttributes["flats.$bhk.furnish_type"] = "$bhk - Furnish Type";
            $customAttributes["flats.$bhk.built_up_area"] = "$bhk - Built Up Area";
            $customAttributes["flats.$bhk.carpet_area"] = "$bhk - Carpet Area";
            $customAttributes["flats.$bhk.total_floors"] = "$bhk - Total Floors";
            $customAttributes["flats.$bhk.property_on_floor"] = "$bhk - Property On Floor";
            $customAttributes["flats.$bhk.price_per_sqft"] = "$bhk - Price Per Sqft";
            $customAttributes["flats.$bhk.expected_price"] = "$bhk - Expected Price";
        }

        $validator = Validator::make($request->all(), $rules, [
            'required' => 'The :attribute field is required.',
            'integer' => 'The :attribute must be an integer.',
            'numeric' => 'The :attribute must be a number.',
            'min' => 'The :attribute must be at least :min.',
        ], $customAttributes);

        if ($validator->fails()) {
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        DB::beginTransaction();
        try {
            $user = auth()->user();
            $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

            if (!$property) {
                return response()->json(['error' => 'Property not found for editing'], 404);
            }

            $input = $request->all();
            if ($request->has('name') && !empty($request->name)) {
                $slugBase = Str::slug($request->name);
                $slug = $slugBase;

                if ($property->exists) {
                    if ($property->name !== $request->name) {
                        $counter = 1;
                        while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                            $slug = $slugBase . '-' . $counter;
                            $counter++;
                        }
                    } else {
                        $slug = $property->slug;
                    }
                } else {
                    $counter = 1;
                    while (Property::where('slug', $slug)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                }
                $input['slug'] = $slug;
            }

            $input['status'] = 0;
            $input['price'] = $request->expected_price;
            $input['user_id'] = $user->id;

            $property->fill($input)->save();

            // Process facilities and distances
            $facilities = $request->input('facilities', []);
            $distances = $request->input('distances', []);
            $distance_units = $request->input('distance_units', []);

            Facility_Distance::where('property_id', $property->id)->delete();

            if (
                is_array($facilities) &&
                is_array($distances) &&
                is_array($distance_units) &&
                count($facilities) === count($distances) &&
                count($facilities) === count($distance_units)
            ) {
                $inserted = [];
                foreach ($facilities as $index => $facilityId) {
                    $key = $facilityId . '_' . $distances[$index] . '_' . $distance_units[$index];
                    if (in_array($key, $inserted)) continue;
                    $inserted[] = $key;

                    Facility_Distance::create([
                        'facility_id' => $facilityId,
                        'property_id' => $property->id,
                        'distance' => $distances[$index],
                        'distance_unit' => $distance_units[$index],
                    ]);
                }
            }

            // Only keep flats for selected BHKs
            $selectedBhk = $request->input('bhk_names', []);
            $allFlats = $request->input('flats', []);
            $filteredFlats = [];
            foreach ($selectedBhk as $bhk) {
                if (!empty($allFlats[$bhk]) && is_array($allFlats[$bhk])) {
                    $filteredFlats[$bhk] = $allFlats[$bhk];
                }
            }

            $this->processFloorPlans($property, $filteredFlats);

            DB::commit();

            session(['property_id' => $property->id]);

            return redirect()->route('agent.properties.step_3', ['id' => $property->id])
                ->with('success', 'Property details saved successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            // Log full error details for debugging
            logger()->error('Property save error: ' . $e->getMessage(), [
                'exception' => $e,
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all(),
                'property_id' => $id,
            ]);
            // Optionally show error message in flash (remove in production)
            return redirect()->back()
                ->with('error', 'An error occurred while saving the property. Please try again. Error: ' . $e->getMessage())
                ->withInput();
        }
    }

    protected function processFloorPlans(Property $property, array $flatsData)
    {
        $existingBhkTypes = $property->floorPlans->pluck('bhk_type')->toArray();
        $newBhkTypes = array_keys($flatsData);

        $toDelete = array_diff($existingBhkTypes, $newBhkTypes);
        if (!empty($toDelete)) {
            FloorPlan::where('property_id', $property->id)
                ->whereIn('name', $toDelete)
                ->delete();
        }

        foreach ($flatsData as $bhkType => $flatData) {
            FloorPlan::updateOrCreate(
                [
                    'property_id' => $property->id,
                    'name' => $bhkType,
                ],
                [
                    'no_of_bathroom'   => $flatData['no_of_bathroom'] ?? null,
                    'no_of_balcony'    => $flatData['no_of_balcony'] ?? null,
                    'covered_parking'  => $flatData['covered_parking'] ?? null,
                    'open_parking'     => $flatData['open_parking'] ?? null,
                    'furnish_type'     => $flatData['furnish_type'] ?? null,
                    'built_up_area'    => $flatData['built_up_area'] ?? null,
                    'carpet_area'      => $flatData['carpet_area'] ?? null,
                    'area_unit'        => $flatData['area_unit'] ?? 'sqft',
                    'total_floors'     => $flatData['total_floors'] ?? null,
                    'property_on_floor' => $flatData['property_on_floor'] ?? null,
                    'size'             => $bhkType,
                    'price_per_sqft'   => $flatData['price_per_sqft'] ?? null,
                    'expected_price'   => $flatData['expected_price'] ?? null,
                ]
            );
        }
    }
    public function pg_step_2($id = null)
    {
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);

        if ($property) {
            $property->meal_offerings = $property->meal_offerings
                ? explode(',', $property->meal_offerings)
                : [];

            $property->common_areas = $property->common_areas
                ? explode(',', $property->common_areas)
                : [];
        }

        return view('agent.properties.pg-plan', ['property' => $property]);
    }
    public function store_pg_step_2(Request $request, $id = null)
    {
        $rules = [
            'name' => 'required',
            'total_beds' => 'required',
            'pg_for' => 'required',
            'suited_for' => 'required',
            'meals_available' => 'required',
            'notice_period' => 'required',
            'lock_in_period' => 'required',
            'common_areas' => 'required|array|min:1',
            'caretaker' => 'required',
            'is_manager_stays' => 'required',
            'veg_allowed' => 'required',
            'opposit_sex_allowed' => 'required',
            'any_time_allowed' => 'required',
            'last_entry_time' => 'required_if:any_time_allowed,no',
            'visitors_allowed' => 'required',
            'guardians_allowed' => 'required',
            'drinking_allowed' => 'required',
            'smoking_allowed' => 'required',
        ];

        // Conditionally add meal_offerings rule
        if ($request->meals_available === 'yes') {
            $rules['meal_offerings'] = 'required|array|min:1';
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $user = auth()->user();

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }

        $input = $request->all();

        // Convert checkbox arrays to comma-separated strings
        if ($request->has('meal_offerings') && is_array($request->meal_offerings)) {
            $input['meal_offerings'] = implode(',', $request->meal_offerings);
        }
        if ($request->has('meal_speciality') && is_array($request->meal_speciality)) {
            $input['meal_speciality'] = implode(',', $request->meal_speciality);
        }

        if ($request->has('common_areas') && is_array($request->common_areas)) {
            $input['common_areas'] = implode(',', $request->common_areas);
        }

        if ($request->has('pg_for') && is_array($request->pg_for)) {
            $input['pg_for'] = implode(',', $request->pg_for);
        }

        if ($request->has('suited_for') && is_array($request->suited_for)) {
            $input['suited_for'] = implode(',', $request->suited_for);
        }

        // Generate slug if name is given
        if ($request->has('name') && !empty($request->name)) {
            $slugBase = Str::slug($request->name);
            $slug = $slugBase;

            if ($property->exists) {
                if ($property->name !== $request->name) {
                    $counter = 1;
                    while (Property::where('slug', $slug)->where('id', '!=', $property->id)->exists()) {
                        $slug = $slugBase . '-' . $counter;
                        $counter++;
                    }
                } else {
                    $slug = $property->slug;
                }
            } else {
                $counter = 1;
                while (Property::where('slug', $slug)->exists()) {
                    $slug = $slugBase . '-' . $counter;
                    $counter++;
                }
            }
            $input['slug'] = $slug;
        }
        $input['caretaker'] = $request->caretaker;
        $input['status'] = 0;
        $input['user_id'] = $user->id;

        // Save property
        $property->fill($input)->save();

        // Store property ID in session
        session(['property_id' => $property->id]);

        return redirect()->route('agent.properties.rooms.step_2', ['id' => $property->id])
            ->with('success', 'Property details saved successfully!');
    }
    public function rooms_step_2($id = null)
    {
        $property = $id
            ? Property::find($id)
            : (session('property_id') ? Property::find(session('property_id')) : null);

        if (!$property) {
            return redirect()->route('agent.properties.step_1')
                ->with('error', 'Property not found. Please start from step 1.');
        }

        // Get rooms ordered by creation date
        $data['property'] = $property;
        $data['rooms'] = Room::where('property_id', $property->id)
            ->orderBy('created_at', 'asc')
            ->get();

        return view('agent.properties.rooms', $data);
    }

    public function store_rooms_step_2(Request $request, $id = null)
    {
        $rules = [
            'rooms' => 'required|array|min:1',
            'rooms.*.type' => 'required|string|in:Private Room,Double Sharing,Triple Sharing,3+ Sharing',
            'rooms.*.total_beds' => 'required|integer|min:1',
            'rooms.*.rent' => 'required|numeric|min:0',
            'rooms.*.deposit' => 'required|numeric|min:0'
        ];

        // Custom validation messages
        $messages = [
            'rooms.*.type.required' => 'Please select a room type for all rooms',
            'rooms.*.total_beds.required' => 'Please specify number of beds for all rooms',
            'rooms.*.rent.required' => 'Please specify rent amount for all rooms',
            'rooms.*.deposit.required' => 'Please specify security deposit for all rooms',
        ];

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {
            // Ensure all rooms have all fields in old input
            $oldRooms = $request->input('rooms', []);
            foreach ($oldRooms as &$room) {
                $room['type'] = $room['type'] ?? '';
                $room['total_beds'] = $room['total_beds'] ?? '';
                $room['rent'] = $room['rent'] ?? '';
                $room['deposit'] = $room['deposit'] ?? '';
                $room['facilities'] = $room['facilities'] ?? [];
            }

            return back()
                ->withErrors($validator)
                ->withInput(['rooms' => $oldRooms])
                ->with('error', 'Please fix the errors in the form.');
        }

        // Get the property
        $property = $id
            ? Property::find($id)
            : (session('property_id') ? Property::find(session('property_id')) : null);

        if (!$property) {
            return redirect()->route('agent.properties.step_1')
                ->with('error', 'Property not found. Please start from step 1.');
        }

        DB::beginTransaction();
        try {
            $existingRoomIds = [];
            $roomsData = $request->input('rooms');

            foreach ($roomsData as $roomData) {
                // Process facilities - ensure it's always stored as JSON
                $facilities = isset($roomData['facilities'])
                    ? (is_array($roomData['facilities'])
                        ? json_encode($roomData['facilities'])
                        : $roomData['facilities'])
                    : json_encode([]);

                $roomData = [
                    'type' => $roomData['type'],
                    'total_beds' => $roomData['total_beds'],
                    'rent' => $roomData['rent'],
                    'deposit' => $roomData['deposit'],
                    'facilities' => $facilities,
                    'property_id' => $property->id,
                ];

                if (!empty($roomData['id'])) {
                    // Update existing room
                    $room = Room::where('property_id', $property->id)
                        ->where('id', $roomData['id'])
                        ->first();

                    if ($room) {
                        $room->update($roomData);
                        $existingRoomIds[] = $room->id;
                    }
                } else {
                    // Create new room
                    $newRoom = Room::create($roomData);
                    $existingRoomIds[] = $newRoom->id;
                }
            }

            // Delete rooms that were removed from the form
            Room::where('property_id', $property->id)
                ->whereNotIn('id', $existingRoomIds)
                ->delete();

            DB::commit();

            return redirect()->route('agent.properties.amenities', ['id' => $property->id])
                ->with('success', 'Rooms information saved successfully!');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()
                ->withInput()
                ->with('error', 'An error occurred while saving rooms. Please try again.');
        }
    }
    public function roomsAmenities($id = null)
    {
        $property = $id
            ? Property::find($id)
            : (session('property_id') ? Property::find(session('property_id')) : null);

        if (!$property) {
            return redirect()->back()->with('error', 'Property not found.');
        }

        $data['property'] = $property;

        $data['attributes'] = Attribute::whereNotIn('name', ['Amenities', 'Amenities For Commercial'])->orderBy('id', 'desc')->get();
        $data['data_attributes'] = $property ? ($property->attributes != null ? json_decode($property->attributes, true) : []) : [];

        return view('agent.properties.rooms-ammenites', $data);
    }
    public function storeAmenities(Request $request, $id = null)
    {
        $property = $id
            ? Property::find($id)
            : (session('property_id') ? Property::find(session('property_id')) : null);

        if (!$property) {
            return back()->with('error', 'Property not found. Please start from step 1.');
        }
        $input = $request->all();
        $input['attributes'] = !empty($input['attributes']) ? json_encode($input['attributes'], true) : json_encode([]);
        $property->fill($input)->save();
        session(['property_id' => $property->id]);

        return redirect()->route('agent.properties.add.info', ['id' => $property->id]);
    }
    public function addInfo($id = null)
    {
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);
        $data['property'] = $property;
        return view('agent.properties.additional-info', $data);
    }
    public function storeAddInfo(Request $request, $id = null)
    {
        $user = auth()->user();

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : new Property());

        if (!$property) {
            return response()->json(['error' => 'Property not found for editing'], 404);
        }

        $input = $request->all();

        $property->fill($input)->save();

        session(['property_id' => $property->id]);

        return redirect()->route('agent.properties.step_3', ['id' => $property->id])
            ->with('success', 'Property details saved successfully!');
    }
    public function step_3($id = null)
    {
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);
        $data['topCities'] = Location::where('status', 1)
            ->where('is_top_city', 1)
            ->orderBy('custom_order', 'asc')
            ->get();

        $data['otherCities'] = Location::where('status', 1)
            ->where('is_top_city', 0)
            ->orderBy('name', 'asc')
            ->get();
        $data['property'] = $property;

        return view('agent.properties.create-step-3', $data);
    }
    public function store_step_3(Request $request, $id = null)
    {
        $rules = [
            'location_id' => 'required',
            'locality' => 'required|string|max:255',
        ];

        $validatedData = $request->validate($rules);

        $city = Location::find($request->location_id)->city;

        $property = $id
            ? Property::find($id)
            : (session('property_id') ? Property::find(session('property_id')) : null);

        if (!$property) {
            return redirect()->back()->with('error', __('Property not found.'));
        }

        $property->fill($request->only(['location_id', 'locality', 'building', 'sub_locality']));
        $property->real_address = $request->building;
        $property->city = $city;
        $property->save();

        session(['property_id' => $property->id]);

        return redirect()
            ->route('agent.properties.step_4', ['id' => $property->id])
            ->with('success', __('Second Step Completed Successfully.'));
    }
    public function step_4($id = null)
    {
        $data['property'] = $id
            ? Property::find($id)
            : (session('property_id') ? Property::find(session('property_id')) : null);

        $data['floor_plans'] = FloorPlan::where('property_id', $data['property']->id)->get();

        $data['rooms'] = Room::where('property_id', $data['property']->id)->get();

        $property = $data['property'];
        $data['selected_category'] = $property ? $property->category_id : null;

        if (!$property) {
            abort(404, 'Property not found');
        }

        if ($property->looking_for === 'for_pg') {
            return view('agent.properties.pg-step-4', $data);
        } elseif ($property->looking_for === 'for_rent') {
            return view('agent.properties.rent-step-4', $data);
        } elseif (in_array($data['selected_category'], [32, 33])) {
            return view('agent.properties.rent-step-4', $data);
        } elseif (auth()->user()->user_type === 'builder/developer' && $property->type === 'residential') {
            return view('agent.properties.builder-step-4', $data);
        } elseif (auth()->user()->user_type === 'builder/developer' && $property->type === 'commercial') {
            return view('agent.properties.create-step-4', $data);
        } else {
            return view('agent.properties.residential-step-4', $data);
        }
    }
    public function rent_store_step_4(Request $request, $id = null)
    {
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);

        $photoIsMissing = !$property?->photo && !$request->hasFile('photo');

        $request->validate([
            'photo' => ($photoIsMissing ? 'required' : 'nullable') . '|image|mimes:jpeg,png,jpg,gif,svg|max:' . config('app.property_image_max_file_size', 10 * 1024),
        ]);

        if ($property) {
            $input = $request->all();

            if ($request->hasFile('photo')) {
                $status = ExtensionValidation($request->file('photo'));
                if (!$status) {
                    return redirect()->back()->with('message', 'File format not supported');
                }
                $input['photo'] = handleMakeImage($request->file('photo'));
            }

            if ($property->created_at->lt(today())) {
                $input['status'] = 3;
            }
            $property->fill($input)->save();

            session()->forget('property_id');
            return redirect()->route('agent.properties.index')
                ->with('success', __('All Steps Completed Successfully.'));
        }

        return redirect()->back()->with('error', __('Property not found.'));
    }
    public function store_step_4(Request $request, $id = null)
    {
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);

        $photoIsMissing = !$property?->photo && !$request->hasFile('photo');

        $request->validate([
            'photo' => ($photoIsMissing ? 'required' : 'nullable') . "|image|mimes:jpeg,png,jpg,gif,svg|max:" . config('app.property_image_max_file_size', 10 * 1024),
        ]);

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);

        if ($property) {
            $input = $request->all();

            if ($request->hasFile('photo')) {
                $status = ExtensionValidation($request->file('photo'));
                if (!$status) {
                    return redirect()->back()->with('message', 'File format not supported');
                }
                $input['photo'] = handleMakeImage($request->file('photo'));
            }
            if ($property->created_at->lt(today())) {
                $input['status'] = 3;
            }

            $property->fill($input)->save();

            if ($request->has('floor_names')) {
                $floorIds = $request->floor_id;
                $floorNames = $request->floor_names;
                $floorImages = $request->file('floor_images');
                $property_id = $id;
                foreach ($floorIds as $index => $id) {
                    if (!empty($id)) {
                        $floorPlan = FloorPlan::find($id);
                        if ($floorPlan) {
                            $floorPlan->name = $floorNames[$index];
                            if (isset($floorImages[$index]) && $floorImages[$index]->isValid()) {
                                $floorPlan->photo = handleUpdateImage($floorImages[$index], $floorPlan->photo);
                            }
                            $floorPlan->save();
                        }
                    } else {
                        $newFloorPlan = new FloorPlan();
                        $newFloorPlan->name = $floorNames[$index];
                        $newFloorPlan->property_id = $property_id;
                        if (isset($floorImages[$index]) && $floorImages[$index]->isValid()) {
                            $newFloorPlan->photo = handleMakeImage($floorImages[$index]);
                        }
                        $newFloorPlan->save();
                    }
                }
            }
            session()->forget('property_id');
            return redirect()->route('agent.properties.index')
                ->with('success', __('All Steps Completed Successfully.'));
        }

        return redirect()->back()->with('error', __('Property not found.'));
    }
    public function builder_store_step_4(Request $request, $id = null)
    {
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);

        $photoIsMissing = !$property?->photo && !$request->hasFile('photo');

        $request->validate([
            'photo' => ($photoIsMissing ? 'required' : 'nullable') . '|image|mimes:jpeg,png,jpg,gif,svg|max:' . config('app.property_image_max_file_size', 10 * 1024),
        ]);

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);

        if ($property) {
            $input = $request->all();
            if ($request->hasFile('photo')) {
                $status = ExtensionValidation($request->file('photo'));
                if (!$status) {
                    return redirect()->back()->with('message', 'File format not supported');
                }
                $input['photo'] = handleMakeImage($request->file('photo'));
            }
            if ($property->created_at->lt(today())) {
                $input['status'] = 3;
            }
            $property->fill($input)->save();
            if ($request->has('floor_id')) {
                $floorIds = $request->floor_id;
                $floorImages = $request->file('floor_images');
                $floorNames = $request->floor_name;

                foreach ($floorIds as $index => $id) {
                    if (!empty($id)) {
                        $floorPlan = FloorPlan::find($id);
                        if ($floorPlan) {
                            // Update floor name if provided
                            if (isset($floorNames[$index])) {
                                $floorPlan->floor_name = $floorNames[$index];
                            }
                            // Update floor image if valid
                            if (isset($floorImages[$index]) && $floorImages[$index]->isValid()) {
                                $floorPlan->photo = handleUpdateImage($floorImages[$index], $floorPlan->photo);
                            }

                            $floorPlan->save();
                        }
                    }
                }
            }

            session()->forget('property_id');
            return redirect()->route('agent.properties.index')
                ->with('success', __('All Steps Completed Successfully.'));
        }
        return redirect()->back()->with('error', __('Property not found.'));
    }
    public function pg_store_step_4(Request $request, $id = null)
    {
        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);

        $photoIsMissing = !$property?->photo && !$request->hasFile('photo');

        $request->validate([
            'photo' => ($photoIsMissing ? 'required' : 'nullable') . '|image|mimes:jpeg,png,jpg,gif,svg|max:' . config('app.property_image_max_file_size', 10 * 1024),
        ]);

        $property = $id ? Property::find($id) : (session('property_id') ? Property::find(session('property_id')) : null);

        if ($property) {
            $input = $request->all();
            if ($request->hasFile('photo')) {
                $status = ExtensionValidation($request->file('photo'));
                if (!$status) {
                    return redirect()->back()->with('message', 'File format not supported');
                }
                $input['photo'] = handleMakeImage($request->file('photo'));
            }
            if ($property->created_at->lt(today())) {
                $input['status'] = 3;
            }
            $property->fill($input)->save();

            if ($request->has('room_id')) {
                $roomIds = $request->room_id;

                foreach ($roomIds as $id) {
                    if (!empty($id)) {
                        $room = Room::find($id);
                        if ($room) {
                            $inputName = 'room_images_' . $id;
                            if ($request->hasFile($inputName) && $request->file($inputName)->isValid()) {
                                $room->photo = handleUpdateImage($request->file($inputName), $room->photo);
                            }
                            $room->save();
                        }
                    }
                }
            }

            session()->forget('property_id');
            return redirect()->route('agent.properties.index')
                ->with('success', __('All Steps Completed Successfully.'));
        }
        return redirect()->back()->with('error', __('Property not found.'));
    }
    public function destroy($id)
    {
        $data = Property::findOrFail($id);
        @unlink('assets/images/' . $data->photo);
        $data->delete();
        // return response()->json($msg);
        return redirect()->route('agent.properties.index')
            ->with('success', __('Property Deleted Successfully.'));
    }
    public function makePremium(Request $request)
    {
        $user = auth()->user();
        $property = Property::where('id', $request->property_id)
            ->where('user_id', $user->id)
            ->first();

        if (!$property) {
            return response()->json([
                'success' => false,
                'message' => 'Property not found.'
            ]);
        }

        if ($user->ad_limit < 1) {
            return response()->json([
                'success' => false,
                'message' => 'Insufficient credits.',
                'availableCredit' => $user->ad_limit,
            ]);
        }
        $user->ad_limit -= 1;
        $user->save();

        $property->is_primium = 1;
        $now = Carbon::now();
        $property->active_date = $now;
        $property->expire_date = $now->copy()->addDays(30);

        $property->save();

        return response()->json([
            'success' => true,
            'message' => 'Property is now premium and active.',
            'availableCredit' => $user->ad_limit,
        ]);
    }
}
