<?php

namespace App\Http\Controllers\Agent;

use App\Http\Controllers\Controller;
use App\Models\PropertyEnquiry;
use Illuminate\Http\Request;
use Datatables;

class ViewContactsController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function datatables(Request $request)
    {
        $saleType = $request->sale_id ?? '';

        $datas = PropertyEnquiry::with('property')
            ->whereHas('property', function ($query) {
                $query->where('user_id', auth()->id())
                    ->where('status', 1);
            })
            
            ->orderBy('id', 'desc');

        if ($saleType) {
            $datas = $datas->whereHas('property', function ($q) use ($saleType) {
                $q->where('looking_for', $saleType);
            });
        }

        // Get all data
        $allData = $datas->get();

        // Total count (for info)
        $totalCount = $allData->count();

        return DataTables::of($allData)
        ->editColumn('created_at', function (PropertyEnquiry $data) {
            return $data->created_at->format('d-m-Y');
        })
        ->addColumn('property_info', function (PropertyEnquiry $data) {
            if ($data->property) {
                $property = $data->property;
                $photoUrl = asset('assets/images/' . $property->photo);
                $name = $property->name ?? 'N/A';
                $price = $property->price ? number_format($property->price) : 'N/A';

                return '
                    <div style="display:flex; align-items:center; gap:10px;">
                        <img src="' . $photoUrl . '" alt="Property Image" width="80" height="60" style="object-fit:cover; border-radius:5px;">
                        <div>
                            <div><strong>' . $name . '</strong></div>
                            <div>₹' . $price . '</div>
                        </div>
                    </div>';
            }
            return '<span class="text-danger">Property Deleted</span>';
        })
        ->addColumn('view_btn', function (PropertyEnquiry $data) {
            if ($data->property) {
                $property = $data->property;
                $routes = [
                    'for_pg' => 'pg.details',
                    'for_rent' => 'rent.details',
                    'for_sell' => 'buy.details',
                    'plot' => 'plot.details',
                    'commercial' => 'commercial.details',
                ];
                $propertyUrl = $routes[$property->looking_for] ?? '#';
                return '<div><a class="btn btn-primary" href="' . route($propertyUrl, $property->slug) . '" target="_blank">View Property</a></div>';
            }
            return '';
        })
        ->rawColumns(['property_info', 'view_btn'])
        ->with([
            'totalCount' => $totalCount,
            'hasPlan'    => !empty(auth()->user()->plan_id),
            'userType'   => auth()->user()->type ?? 'User',
        ])
        ->toJson();

    }


    public function index()
    {
        $allContacts = PropertyEnquiry::with('property')
            ->whereHas('property', function ($query) {
                $query->where('user_id', auth()->id())
                      ->where('status', 1);
            })
            ->where('contact_type', 'view_contact')
            ->orderByDesc('id')
            ->paginate(25);
        
        $data['contactsCount'] = $allContacts->count();
        
        $data['contacts'] = $allContacts;
        return view('agent.view_contacts.index', $data);
    }
}
