<?php

namespace App\Http\Controllers\Frontend;

use App\Models\Blog;
use App\Models\Plan;
use App\Models\Room;
use App\Models\User;
use App\Models\Review;
use App\Models\Category;
use App\Models\Location;
use App\Models\Property;
use App\Models\Pagesetting;
use App\Models\SocialLinks;
use Illuminate\Support\Str;
use App\Models\SeenProperty;
use Illuminate\Http\Request;
use App\Models\PropertyReview;
use App\Http\Controllers\Controller;
use App\Services\PropertyService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Stevebauman\Location\Facades\Location as cLocation;


class BuyController extends Controller
{

    function __construct(private PropertyService $propertyService)
    {}
        

    public function index(Request $request, $location_name = null)
    {
        $data = [];
        if (!$location_name) {
            $city = config('app.search_default_city');
            return redirect()->route('buy.index.location', ['city' => Str::slug($city)]);
        }

        $currentLocation = Str::replace('-', ' ', $location_name);
        
        $data['testimonials'] = Review::orderBy('id', 'desc')->get();
        $data['plans'] = Plan::whereStatus(1)->orderBy('id', 'desc')->get();
        $data['categories'] = Category::whereStatus(1)->orderBy('id', 'desc')->get();

        $data['topCities'] = Location::where('status', 1)
            ->where('is_top_city', 1)
            ->orderBy('custom_order', 'asc')
            ->get();

        $data['otherCities'] = Location::where('status', 1)
            ->where('is_top_city', 0)
            ->orderBy('name', 'asc')
            ->get();

        $data['locations_for_section']  = Location::whereStatus(1)->orderBy('id', 'asc')->take(8)->get();
        $data['properties'] = Property::where('looking_for', '!=', 'for_pg')
            ->where('status', 1)
            ->where('is_invest', 0)
            ->where('looking_for', 'for_sell')
            ->orderBy('id', 'desc')
            ->where('type', 'residential')
            ->whereNotIn('properties.category_id', [33, 32])
            ->limit(6)
            ->get();

        $data['featured_properties'] = Property::whereStatus(1)->whereIsInvest(0)->whereIsFeature(1)->orderBy('id', 'desc')->limit(8)->get();
        $data['blogs'] = Blog::orderBy('id', 'asc')->limit(6)->get();
        $data['ps'] = Pagesetting::first();
        $data['sociallinks'] = SocialLinks::orderBy('id', 'desc')->get();
        $data['home_modules'] = $data['ps']->home_module ? json_decode($data['ps']->home_module, true) : [];
        $data['currentLocation'] = $currentLocation;
        return view('frontend.buy.index', $data);
    }
    private function normalizeBhkKeyword(?string $keyword): ?string
    {
        if (!$keyword) {
            return null;
        }

        // Convert word numbers to digits
        $keywords = [
            'one' => '1',
            'two' => '2',
            'three' => '3',
            'four' => '4',
            'five' => '5',
            'six' => '6',
        ];
        $keyword = str_ireplace(array_keys($keywords), array_values($keywords), $keyword);

        // Remove commas & trim spaces
        $keyword = str_replace(',', '', strtolower(trim($keyword)));

        // Remove extra words like "flat", "home"
        $keyword = preg_replace('/\b(flat|home|apartment|villa)\b/i', '', $keyword);

        // Normalize spacing
        $keyword = preg_replace('/\s+/', ' ', $keyword);

        // Standardize to formats like "1-rk" or "2-bhk"
        $keyword = preg_replace('/\b(\d+)\s?(rk|bhk)\b/i', '$1-$2', $keyword);

        return $keyword ?: null;
    }

    public function search(Request $request)
    {
        $keyword = $request->name;
        $location = $request->location_id;
        $category_id = $request->category_id;
        $no_of_bhk = $request->no_of_bhk;
        $sale_type = $request->sale_type;
        $construction_status = $request->construction_status;
        $minprice = $request->minPrice;
        $maxprice = $request->maxPrice;
        $isTwentyCrPlus = false;
        if ($maxprice == 200000001) {
            $isTwentyCrPlus = true;
            $maxprice = null; // Don't set upper limit
        }
        $user_type = $request->user_type ?: null;
        $developer_checkbox = $request->project ?: null;

        $bhkKeyword = null;
        $localityKeyword = null;

        if ($keyword) {

            $keyword = str_replace(',', '', $keyword);
            $normalizedKeyword = trim(strtoupper(preg_replace('/\s?(bhk|rk)\b/i', '-$1', $keyword)));

            if (stripos($keyword, ' in ') !== false) {
                [$bhkKeyword, $localityKeyword] = array_map('trim', explode(' in ', $keyword, 2));
                $bhkKeyword = preg_replace('/\s?(bhk|rk)\b/i', '-$1', $bhkKeyword);
            } else {
                if (stripos($keyword, 'bhk') !== false || stripos($keyword, 'rk') !== false) {
                    $bhkKeyword = $normalizedKeyword;
                } else {
                    $localityKeyword = $keyword;
                }
            }
        }

        $data['locations'] = Location::whereStatus(1)
            ->orderBy('custom_order')
            ->orderBy('name')
            ->get();

        $query = Property::with(['floorplans'])
            ->select('properties.*')
            ->where('properties.status', 1)
            ->where('properties.looking_for', 'for_sell')
            ->where('type', 'residential')
            ->whereNotIn('properties.category_id', [32, 33]);

        // Location filter
        if ($location) {
            $query->join('locations', 'properties.location_id', '=', 'locations.id')
                ->where('locations.id', '=', $location);
        }

        // Normalize BHK input
        $bhkKeyword = $this->normalizeBhkKeyword($no_of_bhk ?? $bhkKeyword);

        if ($bhkKeyword) {
            $numeric_bhk = preg_replace('/-bhk$/i', '', $bhkKeyword);

            $query->where(function ($q) use ($bhkKeyword, $numeric_bhk) {
                $q->where('properties.no_of_bhk', '=', $bhkKeyword)
                    ->orWhere('properties.no_of_bhk', '=', $numeric_bhk)
                    ->orWhere(function ($q) use ($numeric_bhk) {
                        $q->where('properties.no_of_bhk', 'REGEXP', '[[:<:]]' . $numeric_bhk . '[[:>:]]');
                    })
                    ->orWhereHas('floorplans', function ($q2) use ($bhkKeyword) {
                        $q2->where('size', '=', $bhkKeyword);
                    });
            });

            // Exclude fractional when searching whole numbers
            if (strpos($numeric_bhk, '.') === false) {
                $query->where(function ($q) {
                    $q->whereNull('properties.no_of_bhk')
                        ->orWhere(function ($q2) {
                            $q2->where('properties.no_of_bhk', 'NOT LIKE', '%.5%')
                                ->where('properties.no_of_bhk', 'NOT LIKE', '%.5-bhk%');
                        });
                });
            }
        }
        // Locality keyword or general search
        if ($localityKeyword || $keyword) {
            $query->where(function ($q) use ($keyword, $localityKeyword) {
                $q->where('properties.name', 'LIKE', "%$keyword%")
                    ->orWhere('properties.locality', 'LIKE', "%$localityKeyword%")
                    ->orWhere('properties.sub_locality', 'LIKE', "%$keyword%")
                    ->orWhere('properties.building', 'LIKE', "%$keyword%")
                    ->orWhere('properties.furnish_type', 'LIKE', "%$keyword%");
            });
        }

        // Price filters (convert to integers)
        $minprice = $minprice ? (int)str_replace(',', '', $minprice) : null;
        $maxprice = $maxprice ? (int)str_replace(',', '', $maxprice) : null;

        // Price filters (convert to integers)
        $minprice = $minprice ? (int)str_replace(',', '', $minprice) : null;
        $maxprice = $maxprice ? (int)str_replace(',', '', $maxprice) : null;

        // Check if maxprice is exactly 200000000 (2 crore)
        $isTwoCrPlus = ($maxprice === 200000000);

        if ($minprice !== null || ($maxprice !== null && !$isTwoCrPlus) || $isTwentyCrPlus) {
            $query->where(function ($q) use ($minprice, $maxprice, $isTwentyCrPlus, $isTwoCrPlus) {
                // Match against property price
                if ($minprice !== null) {
                    $q->where('price', '>=', $minprice);
                }
                if ($isTwentyCrPlus || $isTwoCrPlus) {
                    $q->where('price', '>=', ($isTwentyCrPlus ? 200000000 : 0));
                } elseif ($maxprice !== null) {
                    $q->where('price', '<=', $maxprice);
                }

                // OR match against floorplan expected_price
                $q->orWhereHas('floorplans', function ($q2) use ($minprice, $maxprice, $isTwentyCrPlus, $isTwoCrPlus) {
                    if ($minprice !== null) {
                        $q2->where('expected_price', '>=', $minprice);
                    }
                    if ($isTwentyCrPlus || $isTwoCrPlus) {
                        $q2->where('expected_price', '>=', ($isTwentyCrPlus ? 200000000 : 0));
                    } elseif ($maxprice !== null) {
                        $q2->where('expected_price', '<=', $maxprice);
                    }
                });
            });
        }

        $userTypeFilter = $user_type;

        if (!$userTypeFilter && $developer_checkbox === 'yes') {
            $userTypeFilter = 'builder/developer';
        }

        if ($userTypeFilter) {
            $query->join('users', 'users.id', '=', 'properties.user_id')
                ->where('users.user_type', $userTypeFilter);
        }
        // Category filter
        if ($category_id) {
            $query->where('category_id', $category_id);
        }

        // Sale type
        if ($sale_type) {
            $query->where('sale_type', $sale_type);
        }

        // Construction status
        if ($construction_status) {
            $query->where('construction_status', $construction_status);
        }

        $topSlotProperties = $this->propertyService->getSponsoredPropertiesBySlot('buy', $location, 'Top', 2);
        $mediumSlotProperties = $this->propertyService->getSponsoredPropertiesBySlot('buy', $location, 'Medium', 2);

        $data['topSlotProperties'] = $topSlotProperties;
        $data['mediumSlotProperties'] = $mediumSlotProperties;

        $propertyIds = PropertyService::getSponsoredPropertyIds($topSlotProperties, $mediumSlotProperties);
        if(!empty($propertyIds)){
            $query->whereNotIn('properties.id', $propertyIds);
        }
        
        $data['properties'] = $query->orderBy('id', 'desc')->paginate(30)->appends($request->query());

        $city = $location ? Location::find($location)?->name : null;
        $category = $category_id ? Category::find($category_id)?->title : null;

        return view('frontend.buy.search-form', $data)->with('searchData', [
            'keyword' => $keyword,
            'city' => $city,
            'category' => $category,
            'bhk' => $no_of_bhk,
            'minPrice' => $minprice,
            'maxPrice' => $maxprice,
            'saleType' => $sale_type,
            'userType' => $user_type,
            'constructionStatus' => $construction_status,
        ]);
    }

    public function details($slug)
    {
        $data = Property::whereSlug($slug)
            ->with(['facilities', 'facilityDistances.facility', 'enquiries'])
            ->whereStatus(1)
            ->where('properties.looking_for', 'for_sell')
            ->first();
        $isContacted = false;
        if (!$data) {
            return view('frontend.property-not-approved');
        } else {
            if ($data) {
                $isContacted = optional($data->enquiries)
                    ->where('user_id', auth()->id())
                    ->where('property_id', $data->id)
                    ->isNotEmpty();

                $data->isContacted = $isContacted;
            } else {
                $data = (object) ['isContacted' => false];
            }

            $userId = Auth::id();
            if ($userId) {
                SeenProperty::firstOrCreate([
                    'user_id' => $userId,
                    'property_id' => $data->id
                ]);
            }
            if (PropertyReview::wherePropertyId($data->id)->get()) {
                $data['reviews'] = PropertyReview::wherePropertyId($data->id)->whereStatus(1)->get();
            }
            $data['featured_properties'] = Property::whereStatus(1)
                ->whereIsInvest(0)
                ->whereIsFeature(1)
                ->orderBy('id', 'desc')
                ->limit(8)
                ->get();
            $data['data'] = $data;

            if ($data) {
                return view('frontend.buy.property_details', $data);
            } else {
                return view('errors.404');
            }
        }
    }
}
