<?php

namespace App\Http\Controllers\Frontend;

use App\Models\Blog;
use App\Models\Plan;
use App\Models\Room;
use App\Models\User;
use App\Models\Review;
use App\Models\Category;
use App\Models\Location;
use App\Models\Property;
use App\Models\Pagesetting;
use App\Models\SocialLinks;
use Illuminate\Support\Str;
use App\Models\SeenProperty;
use Illuminate\Http\Request;
use App\Models\PropertyReview;
use App\Http\Controllers\Controller;
use App\Services\PropertyService;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Stevebauman\Location\Facades\Location as cLocation;


class PgController extends Controller
{
    public function __construct(private PropertyService $propertyService)
    {
    }
    public function index(Request $request, $location_name = null)
    {
        $data['type'] = $request->type ?? 'buy';
        $data['selected_location_id'] = $request->location_id ?? '';

        // Detect current city
        if (config('app.env') === 'production') {
            $ip = $request->ip();
            $location = cLocation::get($ip);
            $currentLocation = $location->cityName ?? config('app.search_default_city');
        } else {
            $currentLocation = config('app.search_default_city');
        }

        // If visiting root domain `/` (no location in URL), redirect to /buy-real-estate-{city}
        if (!$location_name) {
            $slug = Str::slug($currentLocation); // Requires: use Illuminate\Support\Str;
            return redirect()->route('commercial.index.location', ['location_name' => $slug]);
        }

        // AFFILIATE HANDLING
        $gs = globalSettings();

        // DATA FOR FRONTEND VIEW
        $data['currentLocation'] = $location_name;
        $data['testimonials'] = Review::orderBy('id', 'desc')->get();
        $data['plans'] = Plan::whereStatus(1)->orderBy('id', 'desc')->get();
        $data['categories'] = Category::whereStatus(1)->orderBy('id', 'desc')->get();

        $data['topCities'] = Location::where('status', 1)
            ->where('is_top_city', 1)
            ->orderBy('custom_order', 'asc')
            ->get();

        $data['otherCities'] = Location::where('status', 1)
            ->where('is_top_city', 0)
            ->orderBy('name', 'asc')
            ->get();

        $data['locations_for_section'] = Location::whereStatus(1)
            ->orderBy('id', 'asc')
            ->take(8)
            ->get();

        $data['properties'] = Property::whereStatus(1)
            ->whereIsInvest(0)
            ->orderBy('id', 'desc')
            ->limit(6)
            ->get();

        $data['properties'] = Property::where('looking_for', '!=', 'for_pg')
            ->where('status', 1)
            ->where('is_invest', 0)
            ->whereHas('user', function ($query) {
                $query->where('user_type', '!=', 'builder/developer');
            })
            ->orderBy('id', 'desc')
            ->limit(6)
            ->get();

        $data['blogs'] = Blog::orderBy('id', 'asc')->limit(6)->get();
        $data['ps'] = Pagesetting::first();
        $data['sociallinks'] = SocialLinks::orderBy('id', 'desc')->get();
        $data['home_modules'] = $data['ps']->home_module ? json_decode($data['ps']->home_module, true) : [];

        return view('frontend.pg.index', $data);
    }

    public function search(Request $request)
    {
        $location         = $request->location_id ?: null;
        $keyword          = $request->name;
        $minprice         = $request->minPrice ?: null;
        $maxprice         = $request->maxPrice ?: null;
        $isFiftyKPlus     = false;
        if ($maxprice == 20000) {
            $isFiftyKPlus = true;
            $maxprice = null;
        }
        $meals_available  = $request->meals_available ?: null;
        $pg_for           = $request->gender ?: null; // string or array
        $amenities        = $request->amenities ?: null; // array expected


        $allowedRoomTypes = ['Private Room', 'Double Sharing', 'Triple Sharing', '3+ Sharing'];
        $room_type = in_array($request->room_type, $allowedRoomTypes) ? $request->room_type : null;

        $data['locations'] = Location::whereStatus(1)
            ->orderBy('custom_order', 'asc')
            ->orderBy('name', 'asc')
            ->get();

        $query = Property::with('rooms')
            ->select('properties.*')
            ->where('properties.status', 1)
            ->where('properties.looking_for', 'for_pg');

        if ($location) {
            $query->where('properties.location_id', $location);
        }

        if ($keyword) {
            $query->where(function ($q) use ($keyword) {
                $q->where('properties.name', 'LIKE', "%$keyword%")
                    ->orWhere('properties.sub_locality', 'LIKE', "%$keyword%")
                    ->orWhere('properties.building', 'LIKE', "%$keyword%")
                    ->orWhere('properties.furnish_type', 'LIKE', "%$keyword%")
                    ->orWhere('properties.locality', 'LIKE', "%$keyword%");

                if (in_array(strtolower($keyword), ['pg for girl', 'pg for girls', 'pg girl', 'pg girls','pg for female','pg female','pg for woman','pg woman','pg for women','pg women'])) {
                    $q->orWhere('properties.pg_for', 'LIKE', '%girl%');
                } elseif (in_array(strtolower($keyword), ['pg for boy', 'pg for boys', 'pg boy', 'pg boys','pg for male','pg male','pg for man','pg man','pg for men','pg men'])) {
                    $q->orWhere('properties.pg_for', 'LIKE', '%boy%');
                }
            });
        }

        if ($meals_available) {
            $query->where('properties.meals_available', $meals_available);
        }
        // PG for (handle comma-separated values)
        if ($pg_for) {
            if ($pg_for === 'girls') {
                $query->where('properties.pg_for', 'girls');
            } elseif ($pg_for === 'boys') {
                $query->where('properties.pg_for', 'boys');
            } elseif ($pg_for === 'both') {
                $query->where(function ($q) {
                    $q->where('properties.pg_for', 'girls,boys')
                        ->orWhere('properties.pg_for', 'boys,girls');
                });
            }
        }

        // Room-related filters via relation
        if ($maxprice == 20000) {
            $maxprice = null;
        }

        if ($minprice || $maxprice || $room_type || $amenities) {
            $query->whereHas('rooms', function ($q) use ($minprice, $maxprice, $room_type, $amenities) {
                if ($minprice) {
                    $q->where('rent', '>=', $minprice);
                }

                // Maxprice == 20000 treated as null => no rent upper limit
                if ($maxprice) {
                    $q->where('rent', '<=', $maxprice);
                }

                if ($room_type) {
                    $q->where('type', $room_type);
                }

                if ($amenities) {
                    $amenitiesList = is_array($amenities) ? $amenities : [$amenities];
                    foreach ($amenitiesList as $item) {
                        $q->where('facilities', 'LIKE', "%$item%");
                    }
                }
            });

            if ($room_type) {
                $query->whereDoesntHave('rooms', function ($q) use ($room_type) {
                    $q->where('type', '!=', $room_type);
                });
            }
        }

        
        $topSlotProperties = $this->propertyService->getSponsoredPropertiesBySlot('pg', $location, 'Top', 2);
        $mediumSlotProperties = $this->propertyService->getSponsoredPropertiesBySlot('pg', $location, 'Medium', 2);

        $data['topSlotProperties'] = $topSlotProperties;
        $data['mediumSlotProperties'] = $mediumSlotProperties;

        $propertyIds = PropertyService::getSponsoredPropertyIds($topSlotProperties, $mediumSlotProperties);
        if(!empty($propertyIds)){
            $query->whereNotIn('properties.id',  $propertyIds);
        }

        $data['properties'] = $query
            ->orderBy('properties.id', 'desc')
            ->paginate(30)->appends($request->query());

        $city = $location ? Location::where('id', $location)->value('name') : null;

        return view('frontend.pg.search-form', $data)->with('searchData', [
            'keyword'   => $keyword,
            'city'      => $city,
            'roomType'  => $room_type,
            'minPrice'  => $minprice,
            'maxPrice'  => $maxprice,
        ]);
    }
    public function details($slug)
    {
        $data = Property::whereSlug($slug)
            ->with(['facilities', 'facilityDistances.facility', 'enquiries'])
            ->whereStatus(1)->first();

        $isContacted = false;

        if ($data) {
            $isContacted = optional($data->enquiries)
                ->where('user_id', auth()->id())
                ->where('property_id', $data->id)
                ->isNotEmpty();

            $data->isContacted = $isContacted;
        } else {
            $data = (object) ['isContacted' => false];
        }

        // Store Seen Property (Ensure it's only stored once per user)
        $userId = Auth::id();
        if ($userId) {
            SeenProperty::firstOrCreate([
                'user_id' => $userId,
                'property_id' => $data->id
            ]);
        }
        if (PropertyReview::wherePropertyId($data->id)->get()) {
            $data['reviews'] = PropertyReview::wherePropertyId($data->id)->whereStatus(1)->get();
        }
        $data['featured_properties'] = Property::whereStatus(1)
            ->whereIsInvest(0)
            ->whereIsFeature(1)
            ->orderBy('id', 'desc')
            ->limit(8)
            ->get();
        $data['data'] = $data;

        if ($data) {
            return view('frontend.pg.property_details', $data);
        } else {
            return view('errors.404');
        }
    }
}
