<?php

namespace App\Http\Controllers\Frontend;

use App\Models\Blog;
use App\Models\Plan;
use App\Models\User;
use App\Models\Review;
use App\Models\Category;
use App\Models\Location;
use App\Models\Property;
use App\Models\Pagesetting;
use App\Models\SocialLinks;
use Illuminate\Support\Str;
use App\Models\SeenProperty;
use Illuminate\Http\Request;
use App\Models\PropertyReview;
use App\Http\Controllers\Controller;
use App\Services\PropertyService;
use Illuminate\Support\Facades\Auth;

use Stevebauman\Location\Facades\Location as cLocation;

class PlotController extends Controller
{

    public function __construct(private PropertyService $propertyService)
    {
    }

    public function index(Request $request, $location_name = null)
    {
        $data['selected_location_id'] = $request->location_id ?? '';

        $ip = $request->ip();
        $location = cLocation::get($ip);
        $currentLocation = $location->cityName ?? config('app.search_default_city');

        // If visiting root domain `/` (no location in URL), redirect to /buy-real-estate-{city}
        if (!$location_name) {
            $slug = Str::slug($currentLocation); // Requires: use Illuminate\Support\Str;
            return redirect()->route('commercial.index.location', ['location_name' => $slug]);
        }

        // AFFILIATE HANDLING
        $gs = globalSettings();

        // DATA FOR FRONTEND VIEW
        $data['currentLocation'] = $location_name;
        $data['testimonials'] = Review::orderBy('id', 'desc')->get();
        $data['plans'] = Plan::whereStatus(1)->orderBy('id', 'desc')->get();
        $data['categories'] = Category::whereStatus(1)->orderBy('id', 'desc')->get();

        $data['topCities'] = Location::where('status', 1)
            ->where('is_top_city', 1)
            ->orderBy('custom_order', 'asc')
            ->get();

        $data['otherCities'] = Location::where('status', 1)
            ->where('is_top_city', 0)
            ->orderBy('name', 'asc')
            ->get();

        $data['locations_for_section'] = Location::whereStatus(1)
            ->orderBy('id', 'asc')
            ->take(8)
            ->get();

        $data['properties'] = Property::where('status', 1)
            ->where('is_invest', 0)
            ->whereIn('properties.category_id', [33, 32])
            ->orderBy('id', 'desc')
            ->limit(6)
            ->get();

        $data['featured_properties'] = Property::whereStatus(1)
            ->whereIsInvest(0)
            ->whereIsFeature(1)
            ->orderBy('id', 'desc')
            ->limit(8)
            ->get();

        $data['blogs'] = Blog::orderBy('id', 'asc')->limit(6)->get();
        $data['ps'] = Pagesetting::first();
        $data['sociallinks'] = SocialLinks::orderBy('id', 'desc')->get();
        $data['home_modules'] = $data['ps']->home_module ? json_decode($data['ps']->home_module, true) : [];

        return view('frontend.plot.index', $data);
    }
    public function search(Request $request)
    {
        $keyword = $request->name ?: null;
        $location = $request->location_id ?: null;
        $category_id = $request->category_id ?: null;
        $user_type = $request->user_type ?: null;
        $ownership = $request->ownership ?: null;
        $minprice = $request->minPrice ?: null;
        $maxprice = $request->maxPrice ?: null;
        $originalKeyword = $request->name;

        $data['locations'] = Location::whereStatus(1)->orderBy('custom_order', 'asc')->orderBy('name', 'asc')->get();

        $query = Property::select('properties.*')->where('properties.status', 1)
            ->where('properties.location_id', $location);
        // ->where('properties.expire_date', '>=', now())

        if ($location) {
            $query->join('locations', 'properties.location_id', '=', 'locations.id')
                ->where('locations.id', '=', $location);
        }
        if ($ownership) {
            $query->where('properties.ownership', $ownership);
        }
        if ($category_id) {
            $query->where('properties.category_id', $category_id);
        } else {
            $query->whereIn('properties.category_id', [33, 32]);
        }

        if ($keyword) {
            $query->where(function ($q) use ($keyword, $originalKeyword) {
                $q->where('properties.name', 'LIKE', "%$keyword%")
                    ->orWhere('properties.building', 'LIKE', "%$keyword%")
                    ->orWhere('properties.locality', 'LIKE', "%$keyword%")
                    ->orWhere('properties.sub_locality', 'LIKE', "%$keyword%");

                    if (in_array(strtolower($keyword),['agriculture land','buy agriculture land'])) {
                        $q->orWhere('properties.category_id', 32);
                    } elseif (in_array(strtolower($keyword),['plot','buy plot','plots','buy plots','residential plot','buy residential plot','residential plots','buy residential plots'])) {
                        $q->orWhere('properties.category_id', 33);
                    }
                    
                    
                // properties.plot_space_type
                if (in_array(strtolower($originalKeyword), ['commercial land','buy commercial land','commercial plot','buy commercial plot'])) {
                    $q->orWhere('properties.plot_space_type', '=', "commercial-plots");
                }
                // properties.plot_space_type
                else if (in_array(strtolower($originalKeyword), ['industrial land','buy industrial land'])) {
                    $q->orWhere('properties.plot_space_type', '=', "industrial-lands");
                }
            });
        }

        // Handle numeric filters
        // Price filters (convert to integers)
        $minprice = $minprice ? (int)str_replace(',', '', $minprice) : null;
        $maxprice = $maxprice ? (int)str_replace(',', '', $maxprice) : null;

        // Check if maxprice is exactly 100000000 (for "100000000+" case)
        $isTenLakhPlus = ($maxprice === 100000000);

        if ($minprice !== null || ($maxprice !== null && !$isTenLakhPlus)) {
            $query->where(function ($q) use ($minprice, $maxprice, $isTenLakhPlus) {
                if ($minprice !== null) {
                    $q->where('properties.price', '>=', $minprice);
                }
                if ($maxprice !== null && !$isTenLakhPlus) {
                    $q->where('properties.price', '<=', $maxprice);
                }
            });
        }
        if ($user_type) {
            $query->join('users', 'users.id', '=', 'properties.user_id')
                ->where('users.user_type', $user_type);
        }

        $topSlotProperties = $this->propertyService->getSponsoredPropertiesBySlot('plot', $location, 'Top', 2);
        $mediumSlotProperties = $this->propertyService->getSponsoredPropertiesBySlot('plot', $location, 'Medium', 2);

        $data['topSlotProperties'] = $topSlotProperties;
        $data['mediumSlotProperties'] = $mediumSlotProperties;

        $propertyIds = PropertyService::getSponsoredPropertyIds($topSlotProperties, $mediumSlotProperties);
        if(!empty($propertyIds)){
            $query->whereNotIn('properties.id', $propertyIds);
        }


        // if ($category_id) {
        //     $query->where('properties.category_id', $category_id);
        // }
        $data['properties'] = $query->orderBy('properties.id', 'desc')->paginate(30)->appends($request->query());
        $city = $location ? Location::where('id', $location)->value('name') : null;
        // $category = $category_id ? Category::where('id', $category_id)->value('title') : null;

        return view('frontend.plot.search-form', $data)->with('searchData', [
            'keyword' => $keyword,
            'city' => $city,
            'categoryId' => $category_id,
            'minPrice' => $minprice,
            'maxPrice' => $maxprice,
            'userType' => $user_type,
        ]);
    }
    public function details($slug)
    {
        $data = Property::whereSlug($slug)
            ->with(['facilities', 'facilityDistances.facility', 'enquiries'])
            ->whereStatus(1)->first();

        $isContacted = false;

        if ($data) {
            $isContacted = optional($data->enquiries)
                ->where('user_id', auth()->id())
                ->where('property_id', $data->id)
                ->isNotEmpty();

            $data->isContacted = $isContacted;
        } else {
            $data = (object) ['isContacted' => false];
        }

        // Store Seen Property (Ensure it's only stored once per user)
        $userId = Auth::id();
        if ($userId) {
            SeenProperty::firstOrCreate([
                'user_id' => $userId,
                'property_id' => $data->id
            ]);
        }
        if (PropertyReview::wherePropertyId($data->id)->get()) {
            $data['reviews'] = PropertyReview::wherePropertyId($data->id)->whereStatus(1)->get();
        }
        $data['featured_properties'] = Property::whereStatus(1)
            ->whereIsInvest(0)
            ->whereIsFeature(1)
            ->orderBy('id', 'desc')
            ->limit(8)
            ->get();
        $data['data'] = $data;

        if ($data) {
            return view('frontend.plot.property_details', $data);
        } else {
            return view('errors.404');
        }
    }
}
