<?php

namespace App\Http\Controllers\Frontend;

use App\Models\Plan;
use App\Models\User;
use App\Models\Category;
use App\Models\Facility;
use App\Models\Location;
use App\Models\Property;
use App\Models\Attribute;
use App\Models\Wishlists;
use Illuminate\Http\Request;
use App\Models\PropertyReview;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Models\SeenProperty;
use App\Models\SavedProperty;
use Illuminate\Support\Facades\Auth;


class PropertyController extends Controller
{

    public function listing(Request $request)
    {
        $keyword = $request->name ?: null;
        $category_id = $request->category_id ?: null;
        $location = $request->location_id ?: null;
        $no_of_bhk = $request->no_of_bhk ?: null;
        $sale_type = $request->sale_type ?: null;
        $construction_status = $request->construction_status ?: null;
        $dynamic_type = $request->type ?: null;
        $minprice = $request->min ?: null;
        $maxprice = $request->max ?: null;
        $bhkKeyword = null;
        $localityKeyword = null;
        $keywordParts = [];
        $firstTwoWords = '';

        if ($keyword) {
            $keywords = [
                'one' => '1',
                'two' => '2',
                'three' => '3',
                'four' => '4',
                'five' => '5',
                'six' => '6',
                'seven' => '7',
                'eight' => '8',
                'nine' => '9'
            ];

            foreach ($keywords as $word => $digit) {
                $keyword = str_ireplace($word, $digit, $keyword);
            }

            $keyword = str_replace(',', '', $keyword);
            $keywordParts = array_slice(explode(' ', trim($keyword)), 0, 2);
            $firstTwoWords = implode(' ', $keywordParts);

            $normalizedKeyword = preg_replace('/\s?(bhk|rk)\b/i', '-$1', $keyword);
            $normalizedKeyword = trim(strtoupper($normalizedKeyword));

            if (strpos($keyword, ' in ') !== false) {
                list($bhkKeyword, $localityKeyword) = explode(' in ', $keyword, 2);
                $bhkKeyword = strtoupper(trim(preg_replace('/\s?(bhk|rk)\b/i', '-$1', $bhkKeyword)));
                $localityKeyword = trim($localityKeyword);
            } else {
                if (strpos(strtolower($keyword), 'bhk') !== false || strpos(strtolower($keyword), 'rk') !== false) {
                    $bhkKeyword = strtoupper($normalizedKeyword);
                } else {
                    $localityKeyword = $keyword;
                }
            }
        }

        $data['categories'] = Category::whereStatus(1)->orderBy('id', 'desc')->get();
        $data['locations'] = Location::whereStatus(1)->orderBy('custom_order', 'asc')->orderBy('name', 'asc')->get();

        $query = Property::select('properties.*')->where('properties.status', 1);

        // Handle keyword searches
        if ($bhkKeyword) {
            $query->where('properties.no_of_bhk', 'LIKE', "%$bhkKeyword%");
        }

        if ($location) {
            $query->join('locations', 'properties.location_id', '=', 'locations.id')
                ->where('locations.id', '=', $location);
        }

        if ($firstTwoWords) {
            $query->where(function ($q) use ($firstTwoWords, $keyword, $localityKeyword) {
                $q->where('properties.name', 'LIKE', "%$firstTwoWords%")
                    ->orWhere('properties.furnish_type', 'LIKE', "%$keyword%")
                    ->orWhere('properties.building', 'LIKE', "%$firstTwoWords%")
                    ->orWhere('properties.locality', 'LIKE', "%$localityKeyword%")
                    ->orWhere('properties.sub_locality', 'LIKE', "%$keyword%");
            });
        }

        // Handle numeric filters
        if ($minprice) {
            $query->where('properties.price', '>=', $minprice);
        }
        if ($maxprice) {
            $query->where('properties.price', '<=', $maxprice);
        }
        if ($no_of_bhk) {
            $query->where('properties.no_of_bhk', $no_of_bhk);
        }
        if ($sale_type) {
            $query->where('properties.sale_type', $sale_type);
        }
        if ($construction_status) {
            $query->where('properties.construction_status', $construction_status);
        }

        // Handle dynamic type (priority to explicit category_id if both exist)
        if ($category_id) {
            $query->where('properties.category_id', $category_id);
        } elseif ($dynamic_type) {
            // Check if dynamic_type is a category ID (comma-separated numbers)
            if (preg_match('/^[0-9,]+$/', $dynamic_type)) {
                $categoryIds = explode(',', $dynamic_type);
                $query->whereIn('properties.category_id', $categoryIds);
            }
            // Check if dynamic_type is "for_sale" or "for_rent"
            elseif (in_array($dynamic_type, ['for_sale', 'for_rent'])) {
                $query->where('properties.looking_for', $dynamic_type);
            }
            // Check if dynamic_type is "commercial" or "residential"
            elseif (in_array($dynamic_type, ['commercial', 'residential'])) {
                $query->where('properties.type', $dynamic_type);
            }
        }

        $data['properties'] = $query->orderBy('properties.id', 'desc')->paginate(15);

        $city = $location ? Location::where('id', $location)->value('name') : null;
        $category = $category_id ? Category::where('id', $category_id)->value('title') : null;

        return view('frontend.list', $data)->with('searchData', [
            'keyword' => $keyword,
            'city' => $city,
            'category' => $category,
            'bhk' => $no_of_bhk,
            'minPrice' => $minprice,
            'maxPrice' => $maxprice,
            'saleType' => $sale_type,
            'constructionStatus' => $construction_status,
            'type' => $request->type,
        ]);
    }
    public function getSuggestions(Request $request)
    {
        $keyword = $request->get('query');
        $location = $request->get('location');
        $dynamic_type = $request->get('type');

        // Determine which column to use based on dynamic_type value
        $column_name = null;
        $column_value = null;

        if ($dynamic_type) {
            // Check if dynamic_type is a category ID (comma-separated numbers)
            if (preg_match('/^[0-9,]+$/', $dynamic_type)) {
                $column_name = 'category_id';
                $column_value = $dynamic_type;
            }
            // Check if dynamic_type is "for_sale" or "for_rent"
            elseif (in_array($dynamic_type, ['for_sale', 'for_rent'])) {
                $column_name = 'looking_for';
                $column_value = $dynamic_type;
            }
            // Check if dynamic_type is "commercial" or "residential"
            elseif (in_array($dynamic_type, ['commercial', 'residential'])) {
                $column_name = 'type';
                $column_value = $dynamic_type;
            }
        }

        $normalizedKeyword = null;

        if ($keyword) {
            $keywords = [
                'one' => '1',
                'two' => '2',
                'three' => '3',
                'four' => '4',
                'five' => '5',
                'six' => '6',
                'seven' => '7',
                'eight' => '8',
                'nine' => '9'
            ];

            foreach ($keywords as $word => $digit) {
                $keyword = str_ireplace($word, $digit, $keyword);
            }

            $normalizedKeyword = preg_replace('/\s?(bhk|rk)\b/i', '-$1', $keyword);
            $normalizedKeyword = trim(strtoupper($normalizedKeyword));
        }

        $isBhkMatch = $normalizedKeyword && Property::where('no_of_bhk', 'LIKE', "%$normalizedKeyword%")->exists();

        if ($isBhkMatch) {
            $localities = Property::where('status', 1)
                ->when($location, function ($query) use ($location) {
                    $query->where('city', $location);
                })
                ->when($column_name && $column_value, function ($query) use ($column_name, $column_value) {
                    // For category_id which might be comma-separated
                    if ($column_name === 'category_id') {
                        $categoryIds = explode(',', $column_value);
                        return $query->whereIn('category_id', $categoryIds);
                    }
                    // For other columns
                    return $query->where($column_name, $column_value);
                })
                ->where('no_of_bhk', 'LIKE', "%$normalizedKeyword%")
                ->select('locality')
                ->distinct()
                ->get();

            $suggestions = $localities->map(function ($item) use ($keyword) {
                return [
                    'locality' => $keyword . ' in ' . $item->locality,
                    'match' => 'Locality'
                ];
            });
        } else {
            $query = Property::where('status', 1);

            $query->when($location, function ($query) use ($location) {
                $query->where('city', $location);
            })
                ->when($column_name && $column_value, function ($query) use ($column_name, $column_value) {
                    // For category_id which might be comma-separated
                    if ($column_name === 'category_id') {
                        $categoryIds = explode(',', $column_value);
                        return $query->whereIn('category_id', $categoryIds);
                    }
                    // For other columns
                    return $query->where($column_name, $column_value);
                });

            if (!$keyword) {
                $localities = Property::where('status', 1)
                    ->when($location, function ($query) use ($location) {
                        $query->where('city', $location);
                    })
                    ->when($column_name && $column_value, function ($query) use ($column_name, $column_value) {
                        // For category_id which might be comma-separated
                        if ($column_name === 'category_id') {
                            $categoryIds = explode(',', $column_value);
                            return $query->whereIn('category_id', $categoryIds);
                        }
                        // For other columns
                        return $query->where($column_name, $column_value);
                    })
                    ->select('locality')
                    ->distinct()
                    ->get();

                $suggestions = $localities->map(function ($item) {
                    return [
                        'locality' => $item->locality,
                        'match' => 'Locality'
                    ];
                });
            } else {
                $matchingCategories = Category::where('title', 'LIKE', "%$keyword%")->pluck('id');
                $query->where(function ($query) use ($keyword, $matchingCategories) {
                    $query->where('name', 'LIKE', "%$keyword%")
                        ->orWhere('price', 'LIKE', "%$keyword%")
                        ->orWhere('real_address', 'LIKE', "%$keyword%")
                        ->orWhereIn('category_id', $matchingCategories)
                        ->orWhere('type', 'LIKE', "%$keyword%")
                        ->orWhere('sale_type', 'LIKE', "%$keyword%")
                        ->orWhere('furnish_type', 'LIKE', "%$keyword%")
                        ->orWhere('building', 'LIKE', "%$keyword%")
                        ->orWhere('locality', 'LIKE', "%$keyword%")
                        ->orWhere('sub_locality', 'LIKE', "%$keyword%")
                        ->orWhere('construction_status', 'LIKE', "%$keyword%")
                        ->orWhereHas('user', function ($query) use ($keyword) {
                            $query->where('name', 'LIKE', "%$keyword%");
                        });
                });
                $properties = $query->select(
                    'name as Property',
                    'price as Property-Price',
                    'city as City',
                    'real_address as Address',
                    'amenities as Amenities',
                    'category_id as Property-Category',
                    'type as Property-Type',
                    'building as Building',
                    'locality as Locality',
                    'sub_locality as Sub-Locality',
                    'sale_type as Sale-Type',
                    'no_of_bhk as BHK',
                    'furnish_type as Furnish-Type',
                    'looking_for as Looking-For',
                    'construction_status as Construction-Status'
                )
                    ->distinct()
                    ->get();

                $suggestions = $properties->map(function ($item) use ($keyword, $matchingCategories) {
                    $match = 'Property';
                    foreach (['City', 'Building', 'Locality', 'Sub-Locality', 'Address', 'Amenities', 'Property-Price', 'Property-Category', 'Property-Type', 'Sale-Type', 'BHK', 'Furnish-Type', 'Looking-For', 'Construction-Status'] as $field) {
                        if (isset($item->$field) && stripos($item->$field, $keyword) !== false) {
                            $match = ucfirst(str_replace('_', ' ', $field));
                            break;
                        }
                    }

                    if ($item->category_id && in_array($item->category_id, $matchingCategories->toArray())) {
                        $match = 'Property-Category';
                    }

                    return [
                        'result' => $item,
                        'match' => $match
                    ];
                });
            }
        }

        return response()->json($suggestions);
    }
    public function getlistSuggestions(Request $request)
    {
        $keyword = $request->get('query');
        $location = $request->get('location');
        $looking_for = $request->get('type');

        $data['amenities'] = Attribute::whereStatus(1)->whereName('Amenities')->orderBy('id', 'desc')->get();

        $normalizedKeyword = null;

        if ($keyword) {
            $keywords = [
                'one' => '1',
                'two' => '2',
                'three' => '3',
                'four' => '4',
                'five' => '5',
                'six' => '6',
                'seven' => '7',
                'eight' => '8',
                'nine' => '9'
            ];

            foreach ($keywords as $word => $digit) {
                $keyword = str_ireplace($word, $digit, $keyword);
            }

            $normalizedKeyword = preg_replace('/\s?(bhk|rk)\b/i', '-$1', $keyword);
            $normalizedKeyword = trim(strtoupper($normalizedKeyword));
        }

        $isBhkMatch = $normalizedKeyword && Property::where('no_of_bhk', 'LIKE', "%$normalizedKeyword%")->exists();

        if ($isBhkMatch) {
            $localities = Property::where('status', 1)
                ->when($location, function ($query) use ($location) {
                    $query->where('city', $location);
                })
                ->when($looking_for, function ($query) use ($looking_for) {
                    $query->where('looking_for', $looking_for);
                })
                ->where('no_of_bhk', 'LIKE', "%$normalizedKeyword%")
                ->select('locality')
                ->distinct()
                ->get();

            $suggestions = $localities->map(function ($item) use ($keyword) {
                return [
                    'locality' => $keyword . ' in ' . $item->locality,
                    'match' => 'Locality'
                ];
            });
        } else {
            $query = Property::where('status', 1);

            $query->when($location, function ($query) use ($location) {
                $query->where('city', $location);
            })
                ->when(
                    $looking_for,
                    function ($query) use ($looking_for) {
                        $query->where('looking_for', $looking_for);
                    }
                );

            if (!$keyword) {
                $localities = Property::where('city', $location)
                    ->where('status', 1)
                    ->orwhere('looking_for', $looking_for)
                    ->select('locality')
                    ->distinct()
                    ->get();

                $suggestions = $localities->map(function ($item) {
                    return [
                        'locality' => $item->locality,
                        'match' => 'Locality'
                    ];
                });
            } else {
                $matchingCategories = Category::where('title', 'LIKE', "%$keyword%")->pluck('id');

                $query->where(function ($query) use ($keyword, $matchingCategories) {
                    $query->where('name', 'LIKE', "%$keyword%")
                        ->orWhere('price', 'LIKE', "%$keyword%")
                        ->orWhere('real_address', 'LIKE', "%$keyword%")
                        ->orWhereIn('category_id', $matchingCategories)
                        ->orWhere('type', 'LIKE', "%$keyword%")
                        ->orWhere('sale_type', 'LIKE', "%$keyword%")
                        ->orWhere('furnish_type', 'LIKE', "%$keyword%")
                        ->orWhere('building', 'LIKE', "%$keyword%")
                        ->orWhere('locality', 'LIKE', "%$keyword%")
                        ->orWhere('sub_locality', 'LIKE', "%$keyword%")
                        ->orWhere('construction_status', 'LIKE', "%$keyword%")
                        ->orWhereHas('user', function ($query) use ($keyword) {
                            $query->where('name', 'LIKE', "%$keyword%");
                        });
                });
                $properties = $query->select(
                    'name as Property',
                    'price as Property-Price',
                    'city as City',
                    'real_address as Address',
                    'amenities as Amenities',
                    'category_id as Property-Category',
                    'type as Property-Type',
                    'building as Building',
                    'locality as Locality',
                    'sub_locality as Sub-Locality',
                    'sale_type as Sale-Type',
                    'no_of_bhk as BHK',
                    'furnish_type as Furnish-Type',
                    'looking_for as Looking-For',
                    'construction_status as Construction-Status'
                )
                    ->distinct()
                    ->get();


                $suggestions = $properties->map(function ($item) use ($keyword, $matchingCategories) {
                    $match = 'Property';
                    foreach (['City', 'Building', 'Locality', 'Sub-Locality', 'Address', 'Amenities', 'Property-Price', 'Property-Category', 'Property-Type', 'Sale-Type', 'BHK', 'Furnish-Type', 'Looking-For', 'Construction-Status'] as $field) {
                        if (isset($item->$field) && stripos($item->$field, $keyword) !== false) {
                            $match = ucfirst(str_replace('_', ' ', $field));
                            break;
                        }
                    }

                    if ($item->category_id && in_array($item->category_id, $matchingCategories->toArray())) {
                        $match = 'Property-Category';
                    }

                    return [
                        'result' => $item,
                        'match' => $match
                    ];
                });
            }
        }

        return response()->json($suggestions);
    }
    public function create()
    {
        $user = auth()->user();
        if ($user === null) {
            return redirect()->route('user.otp')->with('warning', 'Please login first!');
        }

        if ($user->plan_id == NULL && $user->status != 2) {
            if ($user->ad_limit == 0) {
                $data['plans'] = Plan::whereStatus(1)->orderBy('id', 'desc')->get();
                return view('frontend.plan', $data);
            }
        }
        $data['categories'] = Category::whereStatus(1)->orderBy('id', 'desc')->get();
        $data['locations'] = Location::whereParentId(NULL)->whereStatus(1)->orderBy('id', 'desc')->get();
        $data['attributes'] = Attribute::whereStatus(1)->orderBy('id', 'desc')->get();

        return view('frontend.submitproperty', $data);
    }

    public function wishlist(Request $request)
    {
        $user = auth()->user();
        if ($user == NULL) {
            return response()->json(['error' => 'Please login first!']);
        }

        if (Wishlists::whereUserId(auth()->id())->wherePropertyId($request->property_id)->exists()) {
            $wishlist = Wishlists::whereUserId(auth()->id())->wherePropertyId($request->property_id)->first();
            $wishlist->delete();
            return response()->json(['error' => 'Directory remove from favourite list']);
        }

        $wishlist = new Wishlists();
        $wishlist->user_id = auth()->id();
        $wishlist->property_id = $request->property_id;
        $wishlist->save();

        return response()->json(['success' => 'Property added into your favourite list.']);
    }

    public function propertyDetails($slug)
    {
        $data = Property::whereSlug($slug)
            ->with(['facilities', 'facilityDistances.facility', 'enquiries'])
            ->whereStatus(1)->first();

        $isContacted = false;

        if ($data) {
            $isContacted = optional($data->enquiries)
                ->where('user_id', auth()->id())
                ->where('property_id', $data->id)
                ->isNotEmpty();

            $data->isContacted = $isContacted;
        } else {
            $data = (object) ['isContacted' => false];
        }

        // Store Seen Property (Ensure it's only stored once per user)
        $userId = Auth::id();
        if ($userId) {
            SeenProperty::firstOrCreate([
                'user_id' => $userId,
                'property_id' => $data->id
            ]);
        }
        if (PropertyReview::wherePropertyId($data->id)->get()) {
            $data['reviews'] = PropertyReview::wherePropertyId($data->id)->whereStatus(1)->get();
        }
        $data['featured_properties'] = Property::whereStatus(1)
            ->whereIsInvest(0)
            ->whereIsFeature(1)
            ->orderBy('id', 'desc')
            ->limit(8)
            ->get();
        $data['data'] = $data;

        if ($data) {
            return view('frontend.property_details', $data);
        } else {
            return view('errors.404');
        }
    }

    public function agents(Request $request)
    {
        $name = $request->agent ? $request->agent : null;

        $data['agents'] = User::when($name, function ($q) use ($name) {
            return $q->where('name', 'LIKE', "%$name%");
        })
            ->whereIsAgent(2)
            ->get();

        return view('frontend.agents', $data);
    }

    public function agentDetails($id)
    {
        $agent = User::whereId($id)->first();
        $data['agent'] = $agent;
        $data['rent_properties'] = $agent->properties()->whereType('for_rent')->orderBy('id', 'desc')->paginate(10);
        $data['buy_properties'] = $agent->properties()->whereType('for_buy')->orderBy('id', 'desc')->paginate(10);
        $data['featured_properties'] = $agent->properties()->whereStatus(1)->whereIsFeature(1)->orderBy('id', 'desc')->limit(4)->get();

        return view('frontend.agentdetails', $data);
    }
    public function saveProperty(Request $request)
    {
        $user = auth()->user();
        if ($user == NULL) {
            return response()->json(['error' => 'Please login first!']);
        }

        if (SavedProperty::whereUserId(auth()->id())->wherePropertyId($request->property_id)->exists()) {
            $wishlist = SavedProperty::whereUserId(auth()->id())->wherePropertyId($request->property_id)->first();
            $wishlist->delete();
            return response()->json(['error' => 'Property removed from saved list']);
        }

        $wishlist = new SavedProperty();
        $wishlist->user_id = auth()->id();
        $wishlist->property_id = $request->property_id;
        $wishlist->save();

        return response()->json(['success' => 'Your Property has been saved.']);
    }
}
