<?php

namespace App\Http\Controllers\Frontend;

use App\Models\Blog;
use App\Models\Plan;
use App\Models\User;
use App\Models\Review;
use App\Models\Category;
use App\Models\Location;
use App\Models\Property;
use App\Models\Pagesetting;
use App\Models\SocialLinks;
use Illuminate\Support\Str;
use App\Models\SeenProperty;
use Illuminate\Http\Request;
use App\Models\PropertyReview;
use App\Http\Controllers\Controller;
use App\Services\PropertyService;
use Illuminate\Support\Facades\Auth;
use Stevebauman\Location\Facades\Location as cLocation;

class RentController extends Controller
{
    public function __construct(private PropertyService $propertyService)
    {
    }

    public function index(Request $request, $location_name = null)
    {
        $data['selected_location_id'] = $request->location_id ?? '';

        if (config('app.env') === 'production') {
            $ip = $request->ip();
            $location = cLocation::get($ip);
            $currentLocation = $location->cityName ?? config('app.search_default_city');
        } else {
            $currentLocation = config('app.search_default_city');
        }

        // If visiting root domain `/` (no location in URL), redirect to /buy-real-estate-{city}
        if (!$location_name) {
            $slug = Str::slug($currentLocation); // Requires: use Illuminate\Support\Str;
            return redirect()->route('commercial.index.location', ['location_name' => $slug]);
        }

        // AFFILIATE HANDLING
        $gs = globalSettings();

        // DATA FOR FRONTEND VIEW
        $data['currentLocation'] = $location_name;
        $data['testimonials'] = Review::orderBy('id', 'desc')->get();
        $data['plans'] = Plan::whereStatus(1)->orderBy('id', 'desc')->get();
        $data['categories'] = Category::whereStatus(1)->orderBy('id', 'desc')->get();

        $data['topCities'] = Location::where('status', 1)
            ->where('is_top_city', 1)
            ->orderBy('custom_order', 'asc')
            ->get();

        $data['otherCities'] = Location::where('status', 1)
            ->where('is_top_city', 0)
            ->orderBy('name', 'asc')
            ->get();

        $data['locations_for_section'] = Location::whereStatus(1)
            ->orderBy('id', 'asc')
            ->take(8)
            ->get();

        $data['properties'] = Property::where('looking_for', '!=', 'for_pg')
            ->where('status', 1)
            ->where('is_invest', 0)
            ->where('type', 'residential')
            ->where('looking_for', 'for_rent')
            ->whereHas('user', function ($query) {
                $query->where('user_type', '!=', 'builder/developer');
            })
            ->orderBy('id', 'desc')
            ->limit(6)
            ->get();

        $data['featured_properties'] = Property::whereStatus(1)
            ->whereIsInvest(0)
            ->whereIsFeature(1)
            ->orderBy('id', 'desc')
            ->limit(8)
            ->get();

        $data['blogs'] = Blog::orderBy('id', 'asc')->limit(6)->get();
        $data['ps'] = Pagesetting::first();
        $data['sociallinks'] = SocialLinks::orderBy('id', 'desc')->get();
        $data['home_modules'] = $data['ps']->home_module ? json_decode($data['ps']->home_module, true) : [];

        return view('frontend.rent.index', $data);
    }

    private function normalizeBhkKeyword(?string $keyword): ?string
    {
        if (!$keyword) {
            return null;
        }

        // Convert word numbers to digits
        $keywords = [
            'one' => '1',
            'two' => '2',
            'three' => '3',
            'four' => '4',
            'five' => '5',
            'six' => '6',
        ];
        $keyword = str_ireplace(array_keys($keywords), array_values($keywords), $keyword);

        // Remove commas & trim spaces
        $keyword = str_replace(',', '', strtolower(trim($keyword)));

        // Remove extra words like "flat", "home"
        $keyword = preg_replace('/\b(flat|home|apartment|villa)\b/i', '', $keyword);

        // Normalize spacing
        $keyword = preg_replace('/\s+/', ' ', $keyword);

        // Standardize to formats like "1-rk" or "2-bhk"
        $keyword = preg_replace('/\b(\d+)\s?(rk|bhk)\b/i', '$1-$2', $keyword);

        return $keyword ?: null;
    }

    public function search(Request $request)
    {
        $keyword = $request->name ?: null;
        $location = $request->location_id ?: null;
        $category_id = $request->category_id ?: null;
        $no_of_bhk = $request->no_of_bhk ?: null;
        $user_type = $request->user_type ?: null;
        $furnish_type = $request->furnish_type ?: null;
        $minprice = $request->minPrice ?: null;
        $maxprice = $request->maxPrice ?: null;

        $bhkKeyword = null;
        $localityKeyword = null;

        if ($keyword) {

            $keyword = str_replace(',', '', $keyword);
            $normalizedKeyword = trim(strtoupper(preg_replace('/\s?(bhk|rk)\b/i', '-$1', $keyword)));

            if (stripos($keyword, ' in ') !== false) {
                [$bhkKeyword, $localityKeyword] = array_map('trim', explode(' in ', $keyword, 2));
                $bhkKeyword = preg_replace('/\s?(bhk|rk)\b/i', '-$1', $bhkKeyword);
            } else {
                if (stripos($keyword, 'bhk') !== false || stripos($keyword, 'rk') !== false) {
                    $bhkKeyword = $normalizedKeyword;
                } else {
                    $localityKeyword = $keyword;
                }
            }
        }

        $data['locations'] = Location::whereStatus(1)
            ->orderBy('custom_order')
            ->orderBy('name')
            ->get();

        $query = Property::with(['floorplans'])
            ->select('properties.*')
            ->where('properties.status', 1)
            ->where('properties.looking_for', 'for_rent')
            ->where('type', 'residential')
            ->whereNotIn('properties.category_id', [32, 33])
            ->whereHas('user', function ($query) {
                $query->where('user_type', '!=', 'builder/developer');
            });

        // Location filter
        if ($location) {
            $query->join('locations', 'properties.location_id', '=', 'locations.id')
                ->where('locations.id', '=', $location);
        }

        // Normalize BHK input
        $bhkKeyword = $this->normalizeBhkKeyword($no_of_bhk ?? $bhkKeyword);

        if ($bhkKeyword) {
            $numeric_bhk = preg_replace('/-bhk$/i', '', $bhkKeyword);

            $query->where(function ($q) use ($bhkKeyword, $numeric_bhk) {
                $q->where('properties.no_of_bhk', '=', $bhkKeyword)
                    ->orWhere('properties.no_of_bhk', '=', $numeric_bhk)
                    ->orWhere(function ($q) use ($numeric_bhk) {
                        $q->where('properties.no_of_bhk', 'REGEXP', '[[:<:]]' . $numeric_bhk . '[[:>:]]');
                    })
                    ->orWhereHas('floorplans', function ($q2) use ($bhkKeyword) {
                        $q2->where('size', '=', $bhkKeyword);
                    });
            });

            // Exclude fractional when searching whole numbers
            if (strpos($numeric_bhk, '.') === false) {
                $query->where(function ($q) {
                    $q->whereNull('properties.no_of_bhk')
                        ->orWhere(function ($q2) {
                            $q2->where('properties.no_of_bhk', 'NOT LIKE', '%.5%')
                                ->where('properties.no_of_bhk', 'NOT LIKE', '%.5-bhk%');
                        });
                });
            }
        }
        // Locality keyword or general search
        if ($localityKeyword || $keyword) {
            $query->where(function ($q) use ($keyword, $localityKeyword) {
                $q->where('properties.name', 'LIKE', "%$keyword%")
                    ->orWhere('properties.locality', 'LIKE', "%$localityKeyword%")
                    ->orWhere('properties.sub_locality', 'LIKE', "%$keyword%")
                    ->orWhere('properties.building', 'LIKE', "%$keyword%")
                    ->orWhere('properties.furnish_type', 'LIKE', "%$keyword%");
            });
        }

        // Price filters (convert to integers)
        $minprice = $minprice ? (int)str_replace(',', '', $minprice) : null;
        $maxprice = $maxprice ? (int)str_replace(',', '', $maxprice) : null;

        // Check if maxprice is exactly 1000000 (for "1000000+" case)
        $isTenLakhPlus = ($maxprice === 1000000);

        if ($minprice !== null || ($maxprice !== null && !$isTenLakhPlus)) {
            $query->where(function ($q) use ($minprice, $maxprice, $isTenLakhPlus) {
                if ($minprice !== null) {
                    $q->where('properties.price', '>=', $minprice);
                }
                if ($maxprice !== null && !$isTenLakhPlus) {
                    $q->where('properties.price', '<=', $maxprice);
                }
            });
        }
        if ($no_of_bhk) {
            $query->where('properties.no_of_bhk', $no_of_bhk);
        }
        if ($user_type) {
            $query->join('users', 'users.id', '=', 'properties.user_id')
                ->where('users.user_type', $user_type);
        }

        if ($category_id) {
            $query->where('properties.category_id', $category_id);
        }
        if ($furnish_type) {
            $query->where('properties.furnish_type', $furnish_type);
        }

        $topSlotProperties = $this->propertyService->getSponsoredPropertiesBySlot('rent', $location, 'Top', 2);
        $mediumSlotProperties = $this->propertyService->getSponsoredPropertiesBySlot('rent', $location, 'Medium', 2);

        $data['topSlotProperties'] = $topSlotProperties;
        $data['mediumSlotProperties'] = $mediumSlotProperties;

        $propertyIds = PropertyService::getSponsoredPropertyIds($topSlotProperties, $mediumSlotProperties);
        if(!empty($propertyIds)){
            $query->whereNotIn('properties.id',  $propertyIds);
        }

        $data['properties'] = $query->orderBy('properties.id', 'desc')->paginate(30)->appends($request->query());
        $city = $location ? Location::where('id', $location)->value('name') : null;
        $category = $category_id ? Category::where('id', $category_id)->value('title') : null;

        return view('frontend.rent.search-form', $data)->with('searchData', [
            'keyword' => $keyword,
            'city' => $city,
            'categoryId' => $category_id,
            'bhk' => $no_of_bhk,
            'minPrice' => $minprice,
            'maxPrice' => $maxprice,
            'userType' => $user_type,
            'furnishType' => $furnish_type
        ]);
    }

    public function details($slug)
    {
        $data = Property::whereSlug($slug)
            ->with(['facilities', 'facilityDistances.facility', 'enquiries'])
            ->whereStatus(1)->first();

        $isContacted = false;

        if ($data) {
            $isContacted = optional($data->enquiries)
                ->where('user_id', auth()->id())
                ->where('property_id', $data->id)
                ->isNotEmpty();

            $data->isContacted = $isContacted;
        } else {
            $data = (object) ['isContacted' => false];
        }

        // Store Seen Property (Ensure it's only stored once per user)
        $userId = Auth::id();
        if ($userId) {
            SeenProperty::firstOrCreate([
                'user_id' => $userId,
                'property_id' => $data->id
            ]);
        }
        if (PropertyReview::wherePropertyId($data->id)->get()) {
            $data['reviews'] = PropertyReview::wherePropertyId($data->id)->whereStatus(1)->get();
        }
        $data['featured_properties'] = Property::whereStatus(1)
            ->whereIsInvest(0)
            ->whereIsFeature(1)
            ->orderBy('id', 'desc')
            ->limit(8)
            ->get();
        $data['data'] = $data;

        if ($data) {
            return view('frontend.rent.property_details', $data);
        } else {
            return view('errors.404');
        }
    }
}
