<?php

namespace App\Http\Controllers\User;

use App\Models\User;
use App\Models\Invest;
use App\Models\Country;
use App\Models\Deposit;
use App\Models\Property;
use App\Models\Withdraw;
use App\Models\Wishlists;
use App\Models\Transaction;
use App\Models\SeenProperty;
use Illuminate\Http\Request;
use App\Classes\GeniusMailer;
use App\Models\ReferralBonus;
use App\Models\SavedProperty;
use App\Models\Generalsetting;
use App\Models\PropertyEnquiry;
use App\Classes\GoogleAuthenticator;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

use App\Models\AdminUserConversation;

class UserController extends Controller
{

    public function __construct()
    {
        $this->middleware('auth');
    }

    public function index(Request $request)
    {
        $userId = auth()->id();
        $data['user'] = Auth::user();
        $currentFilter = request()->input('looking_for', '');

        // PropertyEnquiry with total count
        $contactsQuery = PropertyEnquiry::with('property')
            ->where('user_id', $userId)
            ->whereHas('property', fn($q) => $q->where('status', 1));

        $data['contacts'] = $contactsQuery->distinct('property_id')->orderByDesc('id')->paginate(25);
        $data['contactsTotalCount'] = $contactsQuery->distinct('property_id')->count();
        

        // SavedProperty with total count
        $savedQuery = Wishlists::with('property')
            ->where('user_id', $userId)
            ->whereHas('property', fn($q) => $q->where('status', 1));

        $data['savedProperty'] = $savedQuery->distinct('property_id')->orderByDesc('id')->paginate(25);
        $data['savedPropertyTotalCount'] = $savedQuery->distinct('property_id')->count();

        // SeenProperty with total count
        $seenQuery = SeenProperty::with('property')
            ->where('user_id', $userId)
            ->whereHas('property', fn($q) => $q->where('status', 1));

        $data['seenProperties'] = $seenQuery->distinct('property_id')->orderByDesc('id')->paginate(25);
        $data['seenPropertyTotalCount'] = $seenQuery->distinct('property_id')->count();

        // Other data
        $data['transactions'] = Transaction::whereUserId($userId)->orderBy('id', 'desc')->limit(5)->get();
        $data['total_payouts'] = Withdraw::whereUserId($userId)->whereStatus('completed')->sum('amount');
        $data['total_deposits'] = Deposit::whereUserId($userId)->whereStatus('complete')->sum('amount');
        $data['total_invests'] = Invest::whereUserId($userId)->whereStatus(1)->sum('amount');
        $data['total_transactions'] = Transaction::whereUserId($userId)->sum('amount');
        $data['total_tickets'] = AdminUserConversation::whereUserId($userId)->count();
        $data['total_referral_bonus'] = ReferralBonus::wheretoUserId($userId)->sum('amount');
        $data['properties'] = Property::whereIsInvest(1)->whereStatus(1)->orderByDesc('id')->get();

        return view('user.dashboard', array_merge($data, [
            'currentFilter' => $currentFilter
        ]));
    }
    public function dashboardFilter(Request $request)
    {
        $userId = auth()->id();
        $lookingFor = $request->looking_for;
        $activeTab = $request->active_tab ?? 'contacted-data';

        switch ($activeTab) {
            case 'contacted-data':
                $baseQuery = PropertyEnquiry::with('property')
                    ->where('user_id', $userId)
                    ->whereHas('property', function ($q) use ($lookingFor) {
                        $q->where('status', 1);
                        if ($lookingFor) {
                            $q->where('looking_for', $lookingFor);
                        }
                    });

                $totalCount = $baseQuery->distinct('property_id')->count();
                $paginatedResults = $baseQuery->orderByDesc('id')->distinct('property_id')->paginate(25);

                $html = view('user.partials.contacted_properties', [
                    'contacts' => $paginatedResults,
                    'currentFilter' => $lookingFor,
                    'totalCount' => $totalCount
                ])->render();
                break;

            case 'seen-data':
                $baseQuery = SeenProperty::with('property')
                    ->where('user_id', $userId)
                    ->whereHas('property', function ($q) use ($lookingFor) {
                        $q->where('status', 1);
                        if ($lookingFor) {
                            $q->where('looking_for', $lookingFor);
                        }
                    });

                $totalCount = $baseQuery->distinct('property_id')->count();
                $paginatedResults = $baseQuery->orderByDesc('id')->distinct('property_id')->paginate(25);

                $html = view('user.partials.seen_properties', [
                    'seenProperties' => $paginatedResults,
                    'currentFilter' => $lookingFor,
                    'totalCount' => $totalCount
                ])->render();
                break;

            case 'saved-data':
                $baseQuery = Wishlists::with('property')
                    ->where('user_id', $userId)
                    ->whereHas('property', function ($q) use ($lookingFor) {
                        $q->where('status', 1);
                        if ($lookingFor) {
                            $q->where('looking_for', $lookingFor);
                        }
                    });

                $totalCount = $baseQuery->distinct('property_id')->count();
                $paginatedResults = $baseQuery->orderByDesc('id')->distinct('property_id')->paginate(25);

                $html = view('user.partials.saved_properties', [
                    'savedProperty' => $paginatedResults,
                    'currentFilter' => $lookingFor,
                    'totalCount' => $totalCount
                ])->render();
                break;

            default:
                $html = '<p>No data available</p>';
                $totalCount = 0;
        }

        return response()->json([
            'html' => $html,
            'current_filter' => $lookingFor,
            'total_count' => $totalCount ?? 0
        ]);
    }
    public function transaction(Request $request)
    {
        $user = Auth::user();
        $transactions = Transaction::whereUserId(auth()->id())
            ->when($request->trx_no, function ($query) use ($request) {
                return $query->where('txnid', $request->trx_no);
            })
            ->when($request->type, function ($query) use ($request) {
                if ($request->type != 'all') {
                    return $query->where('type', $request->type);
                } else {
                }
            })
            ->whereUserId(auth()->id())->orderBy('id', 'desc')->paginate(20);
        return view('user.transactions', compact('user', 'transactions'));
    }
    public function profile()
    {
        $user = auth()->user();
        $countries = Country::whereStatus(1)->get();
        return view('user.profile', compact('user', 'countries'));
    }
    public function profileupdate(Request $request)
    {
        $request->validate([
            'photo' => 'mimes:jpeg,jpg,png,svg',
            'email' => 'unique:users,email,' . Auth::user()->id
        ]);

        $input = $request->all();
        $data = Auth::user();
        if ($file = $request->file('photo')) {
            $name = time() . $file->getClientOriginalName();
            $file->move('assets/images/', $name);
            @unlink('assets/images/' . $data->photo);

            $input['photo'] = $name;

            $input['is_provider'] = 0;
        }

        $data->update($input);

        $gs = Generalsetting::first();
        if ($gs->is_smtp == 1) {
            $data = [
                'to' => $data->email,
                'type' => "profile update",
                'cname' => $data->name,
                'oamount' => "",
                'aname' => "",
                'aemail' => "",
                'wtitle' => "",
            ];

            $mailer = new GeniusMailer();
            $mailer->sendAutoMail($data);
        } else {
            $to = $data->email;
            $subject = "Your profile has been update";
            $msg = "Hello " . $data->name . "!\nYour profile has been updated successfully.\nThank you.";
            $headers = "From: " . $gs->from_name . "<" . $gs->from_email . ">";
            mail($to, $subject, $msg, $headers);
        }

        $msg = 'Successfully updated your profile';
        return redirect()->back()->with('message', $msg);
    }
    public function changePasswordForm()
    {
        return view('user.changepassword');
    }
    public function changePassword(Request $request)
    {

        $request->validate([
            'cpass' => 'required',
            'newpass' => 'required|min:6',
            'renewpass' => 'required|same:newpass',
        ]);

        $user = Auth::user();
        $gs = Generalsetting::first();

        if ($request->cpass) {
            if (Hash::check($request->cpass, $user->password)) {
                if ($request->newpass == $request->renewpass) {
                    $input['password'] = Hash::make($request->newpass);
                } else {
                    return redirect()->back()->with('unsuccess', 'Confirm password does not match.');
                }
            } else {
                return redirect()->back()->with('unsuccess', 'Current password Does not match.');
            }
        }
        $user->update($input);

        if ($gs->is_smtp == 1) {
            $data = [
                'to' => $user->email,
                'type' => "password changed",
                'cname' => $user->name,
                'oamount' => "",
                'aname' => "",
                'aemail' => "",
                'wtitle' => "",
            ];

            $mailer = new GeniusMailer();
            $mailer->sendAutoMail($data);
        } else {
            $to = $user->email;
            $subject = "Your password has been changed";
            $msg = "Hello " . $user->name . "!\nYour password has been changed successfully.\nThank you.";
            $headers = "From: " . $gs->from_name . "<" . $gs->from_email . ">";
            mail($to, $subject, $msg, $headers);
        }

        return redirect()->back()->with('success', 'Password Successfully Changed.');
    }

    public function showTwoFactorForm()
    {
        $gnl = Generalsetting::first();
        $ga = new GoogleAuthenticator();
        $user = auth()->user();
        $secret = $ga->createSecret();
        $qrCodeUrl = $ga->getQRCodeGoogleUrl($user->name . '@' . $gnl->title, $secret);
        $prevcode = $user->tsc;
        $prevqr = $ga->getQRCodeGoogleUrl($user->name . '@' . $gnl->title, $prevcode);

        return view('user.twofactor.index', compact('secret', 'qrCodeUrl', 'prevcode', 'prevqr'));
    }

    public function createTwoFactor(Request $request)
    {
        $user = auth()->user();

        $this->validate($request, [
            'key' => 'required',
            'code' => 'required',
        ]);

        $ga = new GoogleAuthenticator();
        $secret = $request->key;
        $oneCode = $ga->getCode($secret);

        if ($oneCode == $request->code) {
            $user->go = $request->key;
            $user->twofa = 1;
            $user->save();

            return redirect()->back()->with('success', 'Two factor authentication activated');
        } else {
            return redirect()->back()->with('error', 'Something went wrong!');
        }
    }


    public function disableTwoFactor(Request $request)
    {

        $this->validate($request, [
            'code' => 'required',
        ]);

        $user = auth()->user();
        $ga = new GoogleAuthenticator();

        $secret = $user->go;
        $oneCode = $ga->getCode($secret);
        $userCode = $request->code;

        if ($oneCode == $userCode) {

            $user->go = null;
            $user->twofa = 0;

            $user->save();

            return redirect()->back()->with('success', 'Two factor authentication disabled');
        } else {
            return redirect()->back()->with('error', 'Something went wrong!');
        }
    }

    public function username($email)
    {
        if ($data = User::where('email', $email)->first()) {
            return $data->name;
        } else {
            return false;
        }
    }

    public function affilate_code()
    {
        $user = Auth::guard('web')->user();
        return view('user.affilate_code', compact('user'));
    }
    public function contactedProperty()
    {
        $contactedProperties = PropertyEnquiry::with('property')
            ->whereUserId(auth()->id())
            ->whereHas('property', function ($query) {
                $query->where('status', 1);
            })
            ->orderBy('id', 'desc')
            ->get();
        return view('user.contacted-property', compact('contactedProperties'));
    }
    public function seenProperty()
    {
        $seenProperties = SeenProperty::with('property')
            ->whereUserId(auth()->id())
            ->whereHas('property', function ($query) {
                $query->where('status', 1);
            })
            ->orderBy('id', 'desc')
            ->get();

        return view('user.seen-property', compact('seenProperties'));
    }
    public function savedProperty()
    {
        $savedProperties = Wishlists::with('property')
            ->whereUserId(auth()->id())
            ->whereHas('property', function ($query) {
                $query->where('status', 1);
            })
            ->orderBy('id', 'desc')
            ->get();
        return view('user.saved-property', compact('savedProperties'));
    }

    public function recentSearches()
    {
        return view('user.recent-search');
    }

    // AJAX filter for dashboard properties by looking_for
    
}
