<?php

namespace App\Providers;

use App\Models\Font;
use App\Models\PaymentGateway;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\ServiceProvider;
use Illuminate\Pagination\Paginator;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Facades\Validator;

class AppServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        //
    }

    /**
     * Bootstrap any application services.
     *
     * @return void
     */
    public function boot()
    {

        if ($this->app->environment('production') && DB::table('generalsettings')->first()->is_ssl == 1) {
            \URL::forceScheme('https');
        }

        // Cache frequently accessed data
        $generalSettings = Cache::rememberForever('generalsettings', function () {
            return DB::table('generalsettings')->first();
        });
        Config::set('global.gs', $generalSettings);

        $seoTools = Cache::rememberForever('seotools', function () {
            return DB::table('seotools')->first();
        });

        $pages = Cache::rememberForever('pages', function () {
            return DB::table('pages')->whereStatus(1)->get();
        });

        $pageSettings = Cache::rememberForever('pagesettings', function () {
            return DB::table('pagesettings')->first();
        });

        $socialSettings = Cache::rememberForever('socialsettings', function () {
            return DB::table('socialsettings')->first();
        });

        $socialLinks = Cache::rememberForever('social_links', function () {
            return DB::table('social_links')->whereStatus(1)->get();
        });

        $defaultFont = Cache::rememberForever('default_font', function () {
            return Font::where('is_default', '=', 1)->first();
        });

        $defaultCurrency = Cache::rememberForever('default_currency', function () {
            return DB::table('currencies')->where('is_default', '=', 1)->first();
        });

        // Cache default language
        $defaultLanguage = Cache::rememberForever('default_language', function () {
            return DB::table('languages')->where('is_default', '=', 1)->first();
        });

        // Cache top and other locations
        $topCities = Cache::rememberForever('top_cities', function () {
            return DB::table('locations')->where('status', 1)->where('is_top_city', 1)->orderBy('custom_order', 'asc')->get();
        });

        $otherCities = Cache::rememberForever('other_cities', function () {
            return DB::table('locations')->where('status', 1)->where('is_top_city', 0)->orderBy('name', 'asc')->get();
        });

        // Cache specific locations by ID
        $locationCache = Cache::rememberForever('locations_by_id', function () {
            return DB::table('locations')->get()->keyBy('id');
        });

        view()->composer('*', function ($settings) use (
            $seoTools,
            $generalSettings,
            $pages,
            $pageSettings,
            $socialSettings,
            $socialLinks,
            $defaultFont,
            $defaultCurrency,
            $defaultLanguage,
            $topCities,
            $otherCities,
            $locationCache
        ) {
            $settings->with('seo', $seoTools);
            $settings->with('gs', $generalSettings);
            $settings->with('pages', $pages);
            $settings->with('ps', $pageSettings);
            $settings->with('social', $socialSettings);
            $settings->with('sociallinks', $socialLinks);
            $settings->with('default_font', $defaultFont);
            $settings->with('defaultCurrency', $defaultCurrency);

            // Use cached default language
            if (\Request::is('admin') || \Request::is('admin/*')) {
                App::setlocale($defaultLanguage->name);
            } else {
                if (Session::has('language')) {
                    $data = DB::table('languages')->find(Session::get('language'));
                    App::setlocale($data->name);
                } else {
                    App::setlocale($defaultLanguage->name);
                }
            }

            // Use cached locations
            $settings->with('topCities', $topCities);
            $settings->with('otherCities', $otherCities);
            $settings->with('locationsById', $locationCache);

            if (Session::has('currency')) {
                $data = DB::table('currencies')->find(Session::get('currency'));
                $settings->with('currency', $data);
            } else {
                $settings->with('currency', $defaultCurrency);
            }
        });

        Paginator::useBootstrap();
    }
}
