<?php

namespace App\Services;

use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use App\Models\Generalsetting;

class WhatsAppGroupService
{
    protected $apiUrl;
    protected $accessToken;
    protected $channelId;
    protected $whatsappApiStatus;

    public function __construct()
    {
        $whatsappConfig = config('app.whatsapp');
        $this->whatsappApiStatus = $whatsappConfig['enabled'] ?? false;
        $this->channelId = $whatsappConfig['channel_id'] ?? '';
        $this->apiUrl = ($whatsappConfig['url'] ?? '') . '/' . $this->channelId;
        $this->accessToken = $whatsappConfig['key'] ?? '';
    }

    public function sendMessageWithTemplate(string $mobileNumber, string $templateName, ?string $headerImageUrl = null)
    {
        if (!$this->whatsappApiStatus) {
            return ['status' => false, 'message' => 'WhatsApp API is disabled'];
        }
        try {
            $components = [];

            $templatePayload = [
                "name" => $templateName,
                "language" => [
                    "code" => 'en_US'
                ]
            ];

            if (!empty($headerImageUrl)) {
                $components[] = [
                    "type" => "header",
                    "parameters" => [
                        [
                            "type" => "image",
                            "image" => [
                                "link" => $headerImageUrl
                            ]
                        ]
                    ]
                ];
            }

            if (!empty($components)) {
                $templatePayload["components"] = $components;
            }

            $data = [
                "messaging_product" => "whatsapp",
                "recipient_type" => "individual",
                "to" => $mobileNumber,
                "type" => "template",
                "template" => $templatePayload
            ];

            $response = Http::withToken($this->accessToken)->post($this->apiUrl, $data);

            if ($response->successful()) {
                return ['status' => true, 'message' => 'Message sent successfully', 'response' => $response->json()];
            } else {
                Log::error('WhatsApp API Error', ['response' => $response->body()]);
                return ['status' => false, 'message' => 'Failed to send message', 'error' => $response->body()];
            }
        } catch (Exception $e) {
            Log::error('WhatsApp Exception', ['error' => $e->getMessage()]);
            return ['status' => false, 'message' => 'Exception occurred', 'error' => $e->getMessage()];
        }
    }
}
