@props([
    'label' => 'Plot Area',
    'name' => 'plot_area',
    'type' => 'text',
    'placeholder' => '',
    'value' => '',
    'step' => null,
    'unitSelectEnabled' => false,
    'unitName' => 'area_unit',
    'selectedUnit' => old('area_unit', 'sqft'),
    'id' => 'input-' . uniqid(),
    'error' => null, // Added error prop
])

@php
    $unitSelectEnabled = filter_var($unitSelectEnabled, FILTER_VALIDATE_BOOLEAN);
    $unitId = 'unit-selector-' . $id;
    $unitMap = [
        'sqft' => 'Sq. Ft.',
        'sqm' => 'Sq. Mtr.',
    ];
    $dimensionMap = [
        'sqft' => 'Ft',
        'sqm' => 'Mtr',
    ];

    $selectedUnit = in_array($selectedUnit, ['sqft', 'sqm', 'acres']) ? $selectedUnit : 'sqft';
    $inputType = 'text';
    $isNumberField = $type === 'number';
    
    // Get error message either from passed error prop or Laravel's error bag
    $errorMessage = $error ?? $errors->first($name);
    $unitErrorMessage = $errors->first($unitName);
@endphp

<h6 class="my-2">
    {{ __($label) }}
    @if ($attributes->get('data-unit-dependent', false))
        <span data-unit-display>( {{ $unitMap[$selectedUnit] }})</span>
    @endif
</h6>
<div class="form-group">
    <div class="d-flex align-items-center gap-0">
        <input type="{{ $inputType }}" name="{{ $name }}" id="{{ $id }}"
            class="form-control {{ $unitSelectEnabled ? 'w-25' : 'w-50' }} {{ $errorMessage ? 'is-invalid' : '' }}" 
            value="{{ $value }}"
            @if ($step) step="{{ $step }}" @endif 
            placeholder="{{ $placeholder }}"
            @if ($isNumberField) inputmode="numeric" @endif 
            {!! $attributes->merge(['class' => '']) !!}>

        @if ($unitSelectEnabled)
            <select name="{{ $unitName }}" id="{{ $unitId }}" 
                class="form-control w-25 {{ $unitErrorMessage ? 'is-invalid' : '' }}">
                <option value="">{{ __('Select Unit') }}</option>
                <option value="sqft" {{ $selectedUnit == 'sqft' ? 'selected' : '' }}>Sq. Ft.</option>
                <option value="sqm" {{ $selectedUnit == 'sqm' ? 'selected' : '' }}>Sq. Mtr.</option>
            </select>
        @endif
    </div>
    
    {{-- Client-side validation error --}}
    @if ($isNumberField)
        <span id="{{ $id }}-error" class="text-danger ms-2" style="display: none;">
            Only numeric values are allowed.
        </span>
    @endif
    
    {{-- Server-side validation error for main input --}}
    @if ($errorMessage)
        <small class="text-danger">{{ $errorMessage }}</small>
    @endif

    {{-- Server-side validation error for unit select --}}
    @if ($unitSelectEnabled && $unitErrorMessage)
        <small class="text-danger">{{ $unitErrorMessage }}</small>
    @endif
</div>

@if ($unitSelectEnabled || $isNumberField)
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const input = document.getElementById('{{ $id }}');
            @if ($isNumberField)
                const errorSpan = document.getElementById('{{ $id }}-error');

                input.addEventListener('input', function() {
                    const value = input.value;
                    const isValid = /^[0-9]*\.?[0-9]*$/.test(value);

                    if (!isValid) {
                        errorSpan.style.display = 'inline';
                        input.classList.add('is-invalid');
                        input.value = value.replace(/[^0-9.]/g, '');
                    } else {
                        errorSpan.style.display = 'none';
                        input.classList.remove('is-invalid');
                    }
                });

                input.addEventListener('blur', function() {
                    const value = input.value;
                    if (value && !/^[0-9]*\.?[0-9]*$/.test(value)) {
                        errorSpan.style.display = 'inline';
                        input.classList.add('is-invalid');
                    }
                });
            @endif

            @if ($unitSelectEnabled)
                const unitMap = @json($unitMap);
                const dimensionMap = @json($dimensionMap);
                const unitSelect = document.getElementById('{{ $unitId }}');

                function updateUnitDisplays(unit) {
                    const displayUnit = unit || 'sqft';
                    document.querySelectorAll('[data-unit-display]').forEach(el => {
                        el.textContent = `( in ${unitMap[displayUnit]})`;
                    });
                    document.dispatchEvent(new CustomEvent('unitChanged', {
                        detail: {
                            unit: displayUnit,
                            areaUnit: unitMap[displayUnit],
                            dimensionUnit: dimensionMap[displayUnit]
                        }
                    }));
                }

                updateUnitDisplays('{{ $selectedUnit }}');

                if (unitSelect) {
                    unitSelect.addEventListener('change', function() {
                        updateUnitDisplays(this.value);
                    });
                }
            @endif
        });
    </script>
@endif